/*******************************************************************************
 * eGov suite of products aim to improve the internal efficiency,transparency,
 *    accountability and the service delivery of the government  organizations.
 *
 *     Copyright (C) <2015>  eGovernments Foundation
 *
 *     The updated version of eGov suite of products as by eGovernments Foundation
 *     is available at http://www.egovernments.org
 *
 *     This program is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program. If not, see http://www.gnu.org/licenses/ or
 *     http://www.gnu.org/licenses/gpl.html .
 *
 *     In addition to the terms of the GPL license to be adhered to in using this
 *     program, the following additional terms are to be complied with:
 *      1) All versions of this program, verbatim or modified must carry this
 *         Legal Notice.
 *
 *      2) Any misrepresentation of the origin of the material is prohibited. It
 *         is required that all modified versions of this material be marked in
 *         reasonable ways as different from the original version.
 *
 *      3) This license does not grant any rights to any user of the program
 *         with regards to rights under trademark law for use of the trade names
 *         or trademarks of eGovernments Foundation.
 *
 *   In case of any queries, you can reach eGovernments Foundation at contact@egovernments.org.
 ******************************************************************************/
package org.egov.ap.ptis.service;

import org.apache.commons.io.FileUtils;
import org.apache.log4j.Logger;
import org.egov.ap.ptis.model.DemandBill;
import org.egov.commons.CFinancialYear;
import org.egov.commons.dao.FinancialYearDAO;
import org.egov.infra.admin.master.entity.City;
import org.egov.infra.admin.master.service.CityService;
import org.egov.infra.exception.ApplicationRuntimeException;
import org.egov.infra.filestore.entity.FileStoreMapper;
import org.egov.infra.filestore.service.FileStoreService;
import org.egov.infra.reporting.engine.ReportFormat;
import org.egov.infra.reporting.engine.ReportOutput;
import org.egov.infra.reporting.engine.ReportRequest;
import org.egov.infra.reporting.engine.ReportService;
import org.egov.infra.utils.DateUtils;
import org.egov.infstr.services.PersistenceService;
import org.egov.ptis.constants.PropertyTaxConstants;
import org.egov.ptis.domain.dao.property.BasicPropertyDAO;
import org.egov.ptis.domain.entity.property.BasicProperty;
import org.egov.ptis.domain.service.notice.NoticeService;
import org.egov.ptis.domain.service.property.PropertyPersistenceService;
import org.egov.ptis.domain.service.property.PropertyService;
import org.egov.ptis.notice.PtNotice;
import org.egov.ptis.service.DemandBill.DemandBillService;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.procedure.ProcedureCall;
import org.hibernate.procedure.ProcedureOutputs;
import org.hibernate.result.ResultSetOutput;
import org.joda.time.DateTime;
import org.joda.time.format.DateTimeFormat;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.support.TransactionTemplate;

import javax.persistence.EntityManager;
import javax.persistence.ParameterMode;
import javax.persistence.PersistenceContext;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static org.egov.ptis.constants.PropertyTaxConstants.FILESTORE_MODULE_NAME;
import static org.egov.ptis.constants.PropertyTaxConstants.INTEGRATED_BILL;
import static org.egov.ptis.constants.PropertyTaxConstants.NOTICE_TYPE_BILL;
import static org.egov.ptis.constants.PropertyTaxConstants.QUARTZ_BULKBILL_JOBS;
import static org.egov.ptis.constants.PropertyTaxConstants.REPORT_TEMPLATENAME_DEMANDNOTICE_GENERATION;
import static org.egov.ptis.constants.PropertyTaxConstants.REPORT_TEMPLATENAME_INTEGRATED_DEMANDNOTICE_GENERATION;

/**
 * @author subhash
 */
@Service("demandBillService")
@Transactional(readOnly = true)
public class DemandBillServiceImpl implements DemandBillService {

    private static final Logger LOGGER = Logger.getLogger(DemandBillServiceImpl.class);

    private static final String SCHEDULE_II_OF_APM_ACT_1965 = "(Issued Under Section-91 and Rules 29 to 34 of Schedule-II of AP Municipalities Act, 1965)";

    private static final String CORPORATION_ACT_1955 = "(Issued Under Section-266(1) of AP Municipal Corporations Act, 1994(formerly GHMC Act, 1955))";

    private static final String CDMA_AP_GOV_IN = ".cdma.ap.gov.in";

    private static final String TAXATION_RULE_N_ACT_1965 = " u/S 130APM, Act 1965 and Rule 30(3) of Taxation and Finance Rules";

    private static final String DEMAND_BILL_QUERY = "From DemandBill where assessmentNo = ? and createdDate >= ? and isactive = true and isHistory = 'N'";

    private PersistenceService persistenceService;

    private ReportService reportService;

    private BasicPropertyDAO basicPropertyDAO;

    private NoticeService noticeService;

    private FinancialYearDAO financialYearDAO;

    @Autowired
    private CityService cityService;

    @PersistenceContext
    private EntityManager entityManager;

    @Autowired
    private FileStoreService fileStoreService;

    private TransactionTemplate transactionTemplate;

    @Autowired
    private PropertyPersistenceService basicPropertyService;

    @Autowired
    private PropertyService propertyService;

    @SuppressWarnings({"unchecked", "unused"})
    @Override
    @Transactional
    public ReportOutput generateDemandBill(final String assessmentNo, final String billType) {
        InputStream billPDF = null;
        final CFinancialYear financialYear = financialYearDAO.getFinancialYearByDate(new Date());
        DemandBill demandBill = (DemandBill) persistenceService.find(DEMAND_BILL_QUERY, assessmentNo,
                financialYear.getStartingDate());
        if (demandBill == null) {
            final ProcedureCall procedureCall = getCurrentSession()
                    .createStoredProcedureCall("ptdmdbill_generateforassessment");
            procedureCall.registerParameter("assessmentno", String.class, ParameterMode.IN);
            procedureCall.getParameterRegistration("assessmentno").bindValue(assessmentNo);
            procedureCall.registerParameter("billType", String.class, ParameterMode.IN);
            procedureCall.getParameterRegistration("billType").bindValue(billType);
            final ProcedureOutputs procedureOutputs = procedureCall.getOutputs();
            final ResultSetOutput output = (ResultSetOutput) procedureOutputs.getCurrent();
            final String billNumber = (String) output.getSingleResult();
            demandBill = (DemandBill) persistenceService.find("From DemandBill where billnumber = ? ", billNumber);
        }
        ReportOutput reportOutput = new ReportOutput();
        final PtNotice ptnotice = noticeService.getPtNoticeByNoticeNumberAndBillType(demandBill.getBillnumber(),
                Arrays.asList(NOTICE_TYPE_BILL.toUpperCase(), INTEGRATED_BILL.toUpperCase()));
        if (ptnotice == null) {
            ReportRequest reportRequest;
            final BasicProperty basicProperty = basicPropertyDAO.getBasicPropertyByPropertyID(assessmentNo);
            Map<String, Object> reportParams = prepareReportParams(demandBill);
            if (NOTICE_TYPE_BILL.equals(billType)) {
                reportRequest = new ReportRequest(REPORT_TEMPLATENAME_DEMANDNOTICE_GENERATION,
                        demandBill, reportParams);
            } else {
                reportRequest = new ReportRequest(REPORT_TEMPLATENAME_INTEGRATED_DEMANDNOTICE_GENERATION,
                        demandBill, reportParams);
            }
            reportOutput = reportService.createReport(reportRequest);
            if (reportOutput != null && reportOutput.getReportOutputData() != null)
                billPDF = new ByteArrayInputStream(reportOutput.getReportOutputData());
            noticeService.saveNotice(basicProperty.getPropertyForBasicProperty().getApplicationNo(),
                    demandBill.getBillnumber(), billType, basicProperty, billPDF);
        } else {
            final FileStoreMapper fsm = ptnotice.getFileStore();
            final File file = fileStoreService.fetch(fsm, FILESTORE_MODULE_NAME);
            byte[] bFile;
            try {
                bFile = FileUtils.readFileToByteArray(file);
            } catch (final IOException e) {
                throw new ApplicationRuntimeException("Exception while generating demand bill : " + e);
            }
            reportOutput.setReportOutputData(bFile);
            reportOutput.setReportFormat(ReportFormat.PDF);
        }
        return reportOutput;
    }

    private Map<String, Object> prepareReportParams(final DemandBill demandBill) {
        final Map<String, Object> reportParams = new HashMap<>();
        final SimpleDateFormat currentDateFormatter = new SimpleDateFormat("dd/MM/yyyy");
        final City city = (City) persistenceService.find("from City");
        final BasicProperty basicProperty = basicPropertyDAO.getBasicPropertyByPropertyID(demandBill.getAssessmentNo());
        final String owner = basicProperty.getProperty().getPropertyDetail().getPropertyTypeMaster().getType();
        String ownerType = "";
        if ("Vacant Land".equalsIgnoreCase(owner))
            ownerType = "(On Land)";
        final String cityName = city.getPreferences().getMunicipalityName();
        final String districtName = city.getDistrictName();
        String sectionAct;
        String taxationAct;

        final String date = currentDateFormatter.format(new Date());
        final CFinancialYear financialYear = financialYearDAO.getFinancialYearByDate(new Date());
        final String arrearDate = DateTimeFormat.forPattern("dd/MM/yyyy").print(new DateTime().withDayOfMonth(1));
        final String cityGrade = city.getGrade();
        if (cityGrade != null && cityGrade != ""
                && cityGrade.equalsIgnoreCase(PropertyTaxConstants.CITY_GRADE_CORPORATION)) {
            sectionAct = CORPORATION_ACT_1955;
            taxationAct = "";
        } else {
            sectionAct = SCHEDULE_II_OF_APM_ACT_1965;
            taxationAct = TAXATION_RULE_N_ACT_1965;
        }
        reportParams.put("address", basicProperty.getAddress().toString());
        reportParams.put("oldAssessment", basicProperty.getOldMuncipalNum());
        reportParams.put("cityName", cityName);
        reportParams.put("cityUrl", cityService.findAll().get(0).getName().toLowerCase() + CDMA_AP_GOV_IN);
        reportParams.put("districtName", districtName);
        reportParams.put("currDate", date);
        reportParams.put("financialYear", financialYear.getFinYearRange());
        reportParams.put("firstInstallment", demandBill.getFirstInstDetails());
        reportParams.put("secondInstallment", demandBill.getSecondInstDetails());
        reportParams.put("arrears", demandBill.getArrearInstDetails());
        reportParams.put("arrearDate", arrearDate);
        reportParams.put("sectionAct", sectionAct);
        reportParams.put("taxationAct", taxationAct);
        reportParams.put("ownerType", ownerType);
        reportParams.put("superStructure", basicProperty.getProperty().getPropertyDetail().isStructure() ? Boolean.TRUE : Boolean.FALSE);
        Map<String, BigDecimal> collectionValues = demandBill.getCollections();
        reportParams.put("propertyColl", collectionValues.get("propertyColl"));
        reportParams.put("waterColl", collectionValues.get("waterColl"));
        reportParams.put("sewerageColl", collectionValues.get("sewerageColl"));
        reportParams.put("cityGrade", cityGrade);

        return reportParams;
    }

    @Override
    public void printDemandBill(final String assessmentNo) {
        ReportRequest reportRequest = null;
        InputStream billPDF = null;
        ReportOutput reportOutput = new ReportOutput();
        final CFinancialYear financialYear = financialYearDAO.getFinancialYearByDate(new Date());
        final DemandBill demandBill = (DemandBill) persistenceService.find(
                "From DemandBill where assessmentNo = ? and createdDate >= ? ", assessmentNo,
                financialYear.getStartingDate());
        if (demandBill != null) {
            final Map<String, Object> reportParams = prepareReportParams(demandBill);
            reportRequest = new ReportRequest(REPORT_TEMPLATENAME_INTEGRATED_DEMANDNOTICE_GENERATION, demandBill, reportParams);
            reportOutput = reportService.createReport(reportRequest);
            if (reportOutput != null && reportOutput.getReportOutputData() != null)
                billPDF = new ByteArrayInputStream(reportOutput.getReportOutputData());
            final BasicProperty basicProperty = basicPropertyDAO.getBasicPropertyByPropertyID(assessmentNo);
            noticeService.saveNotice(basicProperty.getPropertyForBasicProperty().getApplicationNo(),
                    demandBill.getBillnumber(), INTEGRATED_BILL, basicProperty, billPDF);
        }
    }

    @Override
    @SuppressWarnings("unchecked")
    public void bulkBillGeneration(final Integer modulo, final Integer billsCount) {

        LOGGER.debug("Entered into executeJob" + modulo);

        final Long startTime = System.currentTimeMillis();

        // returns all the property for which bill is created or cancelled
        final Query query = getQuery(modulo, billsCount);

        final List<String> assessmentNumbers = query.list();
        int noOfBillsGenerated = 0;
        LOGGER.info("executeJob" + modulo + " - got " + assessmentNumbers + "indexNumbers for bill generation");
        Long timeTaken = System.currentTimeMillis() - startTime;
        LOGGER.info("executeJob" + modulo + " took " + timeTaken / 1000
                + " secs for BasicProperty selection - BasicProperties = " + assessmentNumbers.size());
        LOGGER.info("executeJob" + modulo + " - Generating bills.....");
        for (final String assessmentNumber : assessmentNumbers) {
            Boolean status = false;
            final TransactionTemplate txTemplate = new TransactionTemplate(transactionTemplate.getTransactionManager());
            txTemplate.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
            try {
                status = txTemplate.execute(result -> {
                    LOGGER.info("Assessment no - " + assessmentNumber);
                    printDemandBill(assessmentNumber);

                    return Boolean.TRUE;
                });
            } catch (final Exception e) {
                status = txTemplate.execute(result -> {
                    final BasicProperty basicProperty = basicPropertyDAO.getBasicPropertyByPropertyID(assessmentNumber);
                    basicProperty.setBillCrtError(e.toString());
                    basicProperty.setIsBillCreated('N');
                    basicPropertyService.update(basicProperty);
                    LOGGER.error("Error in generating demand bill for assessment - " + assessmentNumber
                            + " and executeJob" + modulo, e);
                    return Boolean.FALSE;
                });
            }
            noOfBillsGenerated = status ? noOfBillsGenerated + 1 : noOfBillsGenerated;
        }
        timeTaken = System.currentTimeMillis() - startTime;
        LOGGER.info("executeJob" + modulo + " - " + noOfBillsGenerated + "/" + assessmentNumbers.size()
                + " Bill(s) generated in " + timeTaken / 1000 + " (secs)");
        LOGGER.debug("Exiting from executeJob" + modulo);
    }

    private Query getQuery(final Integer modulo, final Integer billsCount) {
        final StringBuilder queryString = new StringBuilder(200);
        queryString
                .append("select db.assessmentNo From DemandBill db,BasicPropertyImpl bp where db.assessmentNo = bp.upicNo and (bp.isBillCreated='Y' or bp.isIntgBillCreated = 'Y') and bp.active=true ")
                .append(" and db.isHistory = 'N' and db.billnumber not in (select noticeNo from PtNotice) ").append("and MOD(bp.id, ")
                .append(QUARTZ_BULKBILL_JOBS).append(") = :modulo ");
        final Query query = persistenceService.getSession().createQuery(queryString.toString()).setInteger("modulo",
                modulo);
        query.setMaxResults(billsCount);
        return query;
    }

    /**
     * API to make existing DemandBill inactive
     */
    @Override
    public void makeDemandBillInactive(final String assessmentNo) {
        final CFinancialYear financialYear = financialYearDAO.getFinancialYearByDate(new Date());
        final DemandBill demandBill = (DemandBill) persistenceService.find(DEMAND_BILL_QUERY, assessmentNo,
                financialYear.getStartingDate());
        if (demandBill != null) {
            demandBill.setIsactive(false);
            persistenceService.update(demandBill);
        }
    }

    @Override
    public boolean getDemandBillByAssessmentNumber(final String assessmentNo) {
        final CFinancialYear financialYear = financialYearDAO.getFinancialYearByDate(new Date());
        final DemandBill demandBill = (DemandBill) persistenceService.find(DEMAND_BILL_QUERY, assessmentNo,
                financialYear.getStartingDate());
        return demandBill != null ? true : false;
    }

    @Override
    public Map<String, Object> getDemandBillDetails(final BasicProperty basicProperty) {
        final CFinancialYear financialYear = financialYearDAO.getFinancialYearByDate(new Date());
        final DemandBill demandBill = (DemandBill) persistenceService.find(DEMAND_BILL_QUERY, basicProperty.getUpicNo(),
                financialYear.getStartingDate());
        final Map<String, Object> reportParams = new HashMap<String, Object>();
        reportParams.put("billDate", DateUtils.getDefaultFormattedDate(demandBill.getCreatedDate()));
        reportParams.put("billNumber", demandBill.getBillnumber());
        return reportParams;
    }

    public Session getCurrentSession() {
        return entityManager.unwrap(Session.class);
    }

    public PersistenceService getPersistenceService() {
        return persistenceService;
    }

    public void setPersistenceService(final PersistenceService persistenceService) {
        this.persistenceService = persistenceService;
    }

    public ReportService getReportService() {
        return reportService;
    }

    public void setReportService(final ReportService reportService) {
        this.reportService = reportService;
    }

    public BasicPropertyDAO getBasicPropertyDAO() {
        return basicPropertyDAO;
    }

    public void setBasicPropertyDAO(final BasicPropertyDAO basicPropertyDAO) {
        this.basicPropertyDAO = basicPropertyDAO;
    }

    public NoticeService getNoticeService() {
        return noticeService;
    }

    public void setNoticeService(final NoticeService noticeService) {
        this.noticeService = noticeService;
    }

    public FinancialYearDAO getFinancialYearDAO() {
        return financialYearDAO;
    }

    public void setFinancialYearDAO(final FinancialYearDAO financialYearDAO) {
        this.financialYearDAO = financialYearDAO;
    }

    public TransactionTemplate getTransactionTemplate() {
        return transactionTemplate;
    }

    public void setTransactionTemplate(final TransactionTemplate transactionTemplate) {
        this.transactionTemplate = transactionTemplate;
    }

}
