/*
 * eGov suite of products aim to improve the internal efficiency,transparency,
 *    accountability and the service delivery of the government  organizations.
 *
 *     Copyright (C) <2015>  eGovernments Foundation
 *
 *     The updated version of eGov suite of products as by eGovernments Foundation
 *     is available at http://www.egovernments.org
 *
 *     This program is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program. If not, see http://www.gnu.org/licenses/ or
 *     http://www.gnu.org/licenses/gpl.html .
 *
 *     In addition to the terms of the GPL license to be adhered to in using this
 *     program, the following additional terms are to be complied with:
 *
 *         1) All versions of this program, verbatim or modified must carry this
 *            Legal Notice.
 *
 *         2) Any misrepresentation of the origin of the material is prohibited. It
 *            is required that all modified versions of this material be marked in
 *            reasonable ways as different from the original version.
 *
 *         3) This license does not grant any rights to any user of the program
 *            with regards to rights under trademark law for use of the trade names
 *            or trademarks of eGovernments Foundation.
 *
 *   In case of any queries, you can reach eGovernments Foundation at contact@egovernments.org.
 */

package org.egov.ap.ptis.scheduler.service;

import static org.egov.ptis.constants.PropertyTaxConstants.REG_DEPT_WEBSERVICE_PASSWORD;
import static org.egov.ptis.constants.PropertyTaxConstants.REG_DEPT_WEBSERVICE_USERNAME;
import static org.egov.ptis.constants.PropertyTaxConstants.WF_STATE_REGISTRATION_PENDING;

import java.beans.XMLDecoder;
import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.math.BigDecimal;
import java.rmi.RemoteException;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.List;

import org.egov.eis.entity.Employee;
import org.egov.eis.service.AssignmentService;
import org.egov.infra.exception.ApplicationRuntimeException;
import org.egov.infra.workflow.matrix.entity.WorkFlowMatrix;
import org.egov.infra.workflow.service.SimpleWorkflowService;
import org.egov.infstr.services.PersistenceService;
import org.egov.ptis.domain.entity.property.MutationRegistrationDetails;
import org.egov.ptis.domain.entity.property.PropertyMutation;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.nic.RegistrationDataProxy;
import com.nic.model.CmdaBean;
import com.nic.model.DataBean;

/**
 * 
 * @author subhash
 *
 */
@Service
@Transactional(readOnly = true)
public class MutationApplicationSchedulerService {

    @Autowired
    @Qualifier("workflowService")
    private SimpleWorkflowService<PropertyMutation> transferWorkflowService;

    @Autowired
    @Qualifier("propertyMutationService")
    private PersistenceService<PropertyMutation, Long> propertyMutationService;
    
    @Autowired
    private AssignmentService assignmentService;
    
    @Autowired
    @Qualifier("mutationRegistrationService")
    private PersistenceService<MutationRegistrationDetails, Long> mutationRegistrationService;

    @Transactional
    public void updateMutations() {
        RegistrationDataProxy registrationDataProxy = new RegistrationDataProxy();
        List<PropertyMutation> mutations = getRegistrationPendingMutations();
        for (PropertyMutation mutation : mutations) {
            try {
                byte[] ba = registrationDataProxy.getReg_Document_Info(mutation.getBasicProperty().getUpicNo(),
                        REG_DEPT_WEBSERVICE_USERNAME, REG_DEPT_WEBSERVICE_PASSWORD);
                InputStream fis = new ByteArrayInputStream(ba);
                XMLDecoder xmlDecoder = new XMLDecoder(fis);
                CmdaBean cmdaBean = (CmdaBean) xmlDecoder.readObject();
                if (cmdaBean != null && cmdaBean.getStatusmsg().equalsIgnoreCase("SUCCESS")) {
                    MutationRegistrationDetails registrationDetails = getMutationRegistrationDetails(cmdaBean);
                    mutationRegistrationService.persist(registrationDetails);
                    mutation.setMutationRegistrationDetails(registrationDetails);
                    transitionWorkFlow(mutation);
                    propertyMutationService.persist(mutation);
                }
            } catch (RemoteException e) {
                throw new ApplicationRuntimeException(e.getMessage());
            }
        }
    }

    private MutationRegistrationDetails getMutationRegistrationDetails(CmdaBean cmdaBean) {
        DateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd");
        MutationRegistrationDetails registrationDetails = new MutationRegistrationDetails();
        DataBean dataBean = cmdaBean.getDataBean();
        registrationDetails.setAddress(dataBean.getAddress());
        registrationDetails.setBuyer(dataBean.getBuyer_Name());
        registrationDetails.setSeller(dataBean.getSeller_Name());
        registrationDetails.setDepartmentValue(new BigDecimal(dataBean.getMarket_Value()));
        if (dataBean.getReg_Date() != null) {
            try {
                registrationDetails.setDocumentDate(dateFormat.parse(dataBean.getReg_Date()));
            } catch (ParseException e) {
                throw new ApplicationRuntimeException(e.getMessage());
            }
        }
        registrationDetails.setDocumentLink(cmdaBean.getDocumentLink());
        registrationDetails.setDocumentNo(dataBean.getDocument_No());
        registrationDetails.setDoorNo(dataBean.getDoorNo());
        registrationDetails.setEastBoundary(dataBean.getBoundary_East());
        registrationDetails.setWestBoundary(dataBean.getBoundary_West());
        registrationDetails.setNorthBoundary(dataBean.getBoundary_North());
        registrationDetails.setSouthBoundary(dataBean.getBoundary_South());
        registrationDetails.setPartyValue(new BigDecimal(dataBean.getConsiderable_Value()));
        registrationDetails.setPlinthArea(Double.valueOf(dataBean.getPlinth_Area()));
        registrationDetails.setPlotArea(Double.valueOf(dataBean.getPlot_Area()));
        registrationDetails.setTypeOfTransfer(dataBean.getType_of_Transfer());
        registrationDetails
                .setDocumentValue(registrationDetails.getPartyValue().compareTo(registrationDetails.getDepartmentValue()) > 0
                        ? registrationDetails.getPartyValue() : registrationDetails.getDepartmentValue());
        return registrationDetails;
    }

    private void transitionWorkFlow(final PropertyMutation propertyMutation) {
        final DateTime currentDate = new DateTime();
        Employee user = assignmentService
                .getPrimaryAssignmentForPositon(propertyMutation.getCurrentState().getOwnerPosition().getId()).getEmployee();
        final WorkFlowMatrix wfmatrix = transferWorkflowService.getWfMatrix(propertyMutation.getStateType(), null,
                null, propertyMutation.getType(), propertyMutation.getCurrentState().getValue(), null);
        propertyMutation.transition().progressWithStateCopy().withSenderName(user.getUsername() + "::"
                + user.getName())
                .withStateValue(wfmatrix.getNextState()).withDateInfo(currentDate.toDate())
                .withOwner(propertyMutation.getState().getOwnerPosition())
                .withNextAction(wfmatrix.getNextAction());
    }

    private List<PropertyMutation> getRegistrationPendingMutations() {
        List<PropertyMutation> mutations = propertyMutationService.findAllBy(
                "from PropertyMutation p where p.state.nextAction = ?",
                WF_STATE_REGISTRATION_PENDING);
        return mutations;
    }
}
