/**
 * eGov suite of products aim to improve the internal efficiency,transparency,
   accountability and the service delivery of the government  organizations.

    Copyright (C) <2015>  eGovernments Foundation

    The updated version of eGov suite of products as by eGovernments Foundation
    is available at http://www.egovernments.org

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program. If not, see http://www.gnu.org/licenses/ or
    http://www.gnu.org/licenses/gpl.html .

    In addition to the terms of the GPL license to be adhered to in using this
    program, the following additional terms are to be complied with:

        1) All versions of this program, verbatim or modified must carry this
           Legal Notice.

        2) Any misrepresentation of the origin of the material is prohibited. It
           is required that all modified versions of this material be marked in
           reasonable ways as different from the original version.

        3) This license does not grant any rights to any user of the program
           with regards to rights under trademark law for use of the trade names
           or trademarks of eGovernments Foundation.

  In case of any queries, you can reach eGovernments Foundation at contact@egovernments.org.
 */
package org.egov.collection.service;

import java.math.BigDecimal;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

import org.apache.log4j.Logger;
import org.egov.billsaccounting.services.VoucherConstant;
import org.egov.collection.constants.CollectionConstants;
import org.egov.collection.entity.AccountPayeeDetail;
import org.egov.collection.entity.Challan;
import org.egov.collection.entity.ReceiptDetail;
import org.egov.collection.entity.ReceiptHeader;
import org.egov.collection.entity.ReceiptMisc;
import org.egov.collection.entity.ReceiptVoucher;
import org.egov.collection.integration.models.BillReceiptInfo;
import org.egov.collection.integration.services.BillingIntegrationService;
import org.egov.collection.utils.CollectionsNumberGenerator;
import org.egov.collection.utils.CollectionsUtil;
import org.egov.collection.utils.FinancialsUtil;
import org.egov.commons.Bankaccount;
import org.egov.commons.CFinancialYear;
import org.egov.commons.CVoucherHeader;
import org.egov.commons.EgwStatus;
import org.egov.commons.dao.EgwStatusHibernateDAO;
import org.egov.eis.entity.Employee;
import org.egov.eis.entity.Jurisdiction;
import org.egov.eis.service.DesignationService;
import org.egov.eis.service.EmployeeService;
import org.egov.infra.admin.master.entity.Boundary;
import org.egov.infra.admin.master.entity.Department;
import org.egov.infra.admin.master.entity.User;
import org.egov.infra.admin.master.service.DepartmentService;
import org.egov.infra.exception.ApplicationRuntimeException;
import org.egov.infra.validation.exception.ValidationException;
import org.egov.infstr.models.ServiceDetails;
import org.egov.infstr.services.PersistenceService;
import org.egov.model.contra.ContraJournalVoucher;
import org.egov.model.instrument.InstrumentHeader;
import org.egov.model.instrument.InstrumentType;
import org.egov.pims.commons.Designation;
import org.egov.pims.commons.Position;
import org.hibernate.Query;
import org.springframework.beans.factory.annotation.Autowired;

/**
 * Provides services related to receipt header
 */
public class ReceiptHeaderService extends PersistenceService<ReceiptHeader, Long> {

    private static final Logger LOGGER = Logger.getLogger(ReceiptHeaderService.class);
    private CollectionsUtil collectionsUtil;
    private CollectionsNumberGenerator collectionsNumberGenerator;
    private FinancialsUtil financialsUtil;
    private PersistenceService persistenceService;

    @Autowired
    private DesignationService designationService;
    @Autowired
    private EmployeeService employeeService;
    @Autowired
    private DepartmentService departmentService;
    @Autowired
    private EgwStatusHibernateDAO egwStatusDAO;

    /**
     * @param statusCode
     *            Status code of receipts to be fetched. If null or ALL, then
     *            receipts with all statuses are fetched
     * @param userName
     *            User name of the user who has created the receipts. If null or
     *            ALL, then receipts of all users are fetched
     * @param counterId
     *            Counter id on which the receipts were created. If negative,
     *            then receipts from all counters are fetched
     * @param serviceCode
     *            Service code for which the receipts were created. If null or
     *            ALL, then receipts of all billing services are fetched
     * @return List of all receipts created by given user from given counter id
     *         and having given status
     */
    public List<ReceiptHeader> findAllByStatusUserCounterService(final String statusCode, final Long positionId,
            final Integer counterId, final String serviceCode) {
        final StringBuilder query = new StringBuilder(
                "select receipt from org.egov.collection.entity.ReceiptHeader receipt where 1=1");

        final boolean allStatuses = statusCode == null || statusCode.equals(CollectionConstants.ALL);
        final boolean allCounters = counterId == null || counterId < 0;
        final boolean allPositions = positionId == null || positionId.equals(CollectionConstants.ALL);
        final boolean allServices = serviceCode == null || serviceCode.equals(CollectionConstants.ALL);

        int argCount = 0;
        argCount += allStatuses ? 0 : 1;
        argCount += allCounters ? 0 : 1;
        argCount += allPositions ? 0 : 1;
        argCount += allServices ? 0 : 1;
        final Object arguments[] = new Object[argCount];

        argCount = 0;
        if (!allStatuses) {
            query.append(" and receipt.status.code = ?");
            arguments[argCount++] = statusCode;
        }

        if (!allPositions) {
            query.append(" and receipt.state.ownerPosition.id = ?");
            arguments[argCount++] = positionId;
        }

        if (!allCounters) {
            query.append(" and receipt.location.id = ?");
            arguments[argCount++] = counterId;
        }

        if (!allServices) {
            query.append(" and receipt.service.code = ?");
            arguments[argCount++] = serviceCode;
        }

        query.append(" order by receipt.createdDate desc");

        return findAllBy(query.toString(), arguments);
    }

    /**
     * This method returns the internal reference numbers corresponding to the
     * instrument type.
     *
     * @param entity
     *            an instance of <code>ReceiptHeader</code>
     * @return a <code>List</code> of strings , each representing the internal
     *         reference numbers for each instrument type for the given receipt
     */
    public List<String> generateInternalReferenceNo(final ReceiptHeader entity) {
        final CFinancialYear financialYear = collectionsUtil.getFinancialYearforDate(entity.getCreatedDate());
        final CFinancialYear currentFinancialYear = collectionsUtil.getFinancialYearforDate(new Date());

        return collectionsNumberGenerator.generateInternalReferenceNumber(entity, financialYear, currentFinancialYear);
    }

    /**
     * This method is called for voucher creation into the financial system. For
     * each receipt created in the collections module, a voucher is created.
     *
     * @param receiptHeader
     *            Receipt header for which the pre-approval voucher is to be
     *            created
     * @param receiptBulkUpload
     * @return The created voucher
     */

    protected CVoucherHeader createVoucher(final ReceiptHeader receiptHeader, final Boolean receiptBulkUpload) {
        final HashMap<String, Object> headerdetails = new HashMap<String, Object>();
        final List<HashMap<String, Object>> accountCodeList = new ArrayList<HashMap<String, Object>>();
        final List<HashMap<String, Object>> subledgerList = new ArrayList<HashMap<String, Object>>();
        String fundCode = null, fundsourceCode = null, departmentCode = null;
        Boolean isVoucherApproved = Boolean.FALSE;

        if (receiptHeader.getService().getIsVoucherApproved() != null)
            isVoucherApproved = receiptHeader.getService().getIsVoucherApproved();

        final ReceiptMisc receiptMisc = receiptHeader.getReceiptMisc();
        if (receiptMisc.getFund() != null)
            fundCode = receiptMisc.getFund().getCode();
        if (receiptMisc.getFundsource() != null)
            fundsourceCode = receiptMisc.getFundsource().getCode();
        if (receiptMisc.getDepartment() != null)
            departmentCode = receiptMisc.getDepartment().getCode();

        for (final InstrumentHeader instrumentHeader : receiptHeader.getReceiptInstrument())
            if (instrumentHeader.getInstrumentType().getType().equals(CollectionConstants.INSTRUMENTTYPE_CASH)
                    || instrumentHeader.getInstrumentType().getType().equals(CollectionConstants.INSTRUMENTTYPE_BANK)) {
                headerdetails.put(VoucherConstant.VOUCHERNAME, CollectionConstants.FINANCIAL_RECEIPTS_VOUCHERNAME);
                headerdetails.put(VoucherConstant.VOUCHERTYPE, CollectionConstants.FINANCIAL_RECEIPTS_VOUCHERTYPE);
            } else if (instrumentHeader.getInstrumentType().getType().equals(CollectionConstants.INSTRUMENTTYPE_CHEQUE)
                    || instrumentHeader.getInstrumentType().getType().equals(CollectionConstants.INSTRUMENTTYPE_DD)
                    || instrumentHeader.getInstrumentType().getType().equals(CollectionConstants.INSTRUMENTTYPE_CARD)
                    || instrumentHeader.getInstrumentType().getType().equals(CollectionConstants.INSTRUMENTTYPE_ONLINE))
                if (collectionsUtil.getAppConfigValue(CollectionConstants.MODULE_NAME_COLLECTIONS_CONFIG,
                        CollectionConstants.APPCONFIG_VALUE_RECEIPTVOUCHERTYPEFORCHEQUEDDCARD).equals(
                                CollectionConstants.FINANCIAL_JOURNALVOUCHER_VOUCHERTYPE)) {
                    headerdetails.put(VoucherConstant.VOUCHERNAME,
                            CollectionConstants.FINANCIAL_JOURNALVOUCHER_VOUCHERNAME);
                    headerdetails.put(VoucherConstant.VOUCHERTYPE,
                            CollectionConstants.FINANCIAL_JOURNALVOUCHER_VOUCHERTYPE);
                } else {
                    headerdetails.put(VoucherConstant.VOUCHERNAME, CollectionConstants.FINANCIAL_RECEIPTS_VOUCHERNAME);
                    headerdetails.put(VoucherConstant.VOUCHERTYPE, CollectionConstants.FINANCIAL_RECEIPTS_VOUCHERTYPE);
                }
        headerdetails.put(VoucherConstant.DESCRIPTION, CollectionConstants.FINANCIAL_VOUCHERDESCRIPTION);
        if (receiptHeader.getVoucherDate() == null)
            headerdetails.put(VoucherConstant.VOUCHERDATE, new Date());
        else
            headerdetails.put(VoucherConstant.VOUCHERDATE, receiptHeader.getVoucherDate());

        if (receiptHeader.getVoucherNum() != null && !receiptHeader.getVoucherNum().equals(""))
            headerdetails.put(VoucherConstant.VOUCHERNUMBER, receiptHeader.getVoucherNum());

        headerdetails.put(VoucherConstant.FUNDCODE, fundCode);
        headerdetails.put(VoucherConstant.DEPARTMENTCODE, departmentCode);
        headerdetails.put(VoucherConstant.FUNDSOURCECODE, fundsourceCode);
        headerdetails.put(VoucherConstant.MODULEID, CollectionConstants.COLLECTIONS_EG_MODULES_ID);
        headerdetails.put(VoucherConstant.SOURCEPATH,
                CollectionConstants.RECEIPT_VIEW_SOURCEPATH + receiptHeader.getId());

        Set<ReceiptDetail> receiptDetailSet = new LinkedHashSet<ReceiptDetail>(0);

        /**
         * Aggregate Amount in case of bill based receipt for account codes
         * appearing more than once in receipt details
         */
        if (receiptHeader.getReceipttype() == 'B')
            receiptDetailSet = aggregateDuplicateReceiptDetailObject(new ArrayList<ReceiptDetail>(
                    receiptHeader.getReceiptDetails()));
        else
            receiptDetailSet = receiptHeader.getReceiptDetails();

        for (final ReceiptDetail receiptDetail : receiptDetailSet)
            if (receiptDetail.getCramount().compareTo(BigDecimal.ZERO) != 0
            || receiptDetail.getDramount().compareTo(BigDecimal.ZERO) != 0) {

                final HashMap<String, Object> accountcodedetailsHashMap = new HashMap<String, Object>();
                accountcodedetailsHashMap.put(VoucherConstant.GLCODE, receiptDetail.getAccounthead().getGlcode());

                accountcodedetailsHashMap.put(VoucherConstant.DEBITAMOUNT,
                        receiptDetail.getDramount().compareTo(BigDecimal.ZERO) == 0 ? 0 : receiptDetail.getDramount());
                accountcodedetailsHashMap.put(VoucherConstant.CREDITAMOUNT,
                        receiptDetail.getCramount().compareTo(BigDecimal.ZERO) == 0 ? 0 : receiptDetail.getCramount());
                if (receiptDetail.getFunction() != null)
                    accountcodedetailsHashMap.put(VoucherConstant.FUNCTIONCODE, receiptDetail.getFunction().getCode());
                accountCodeList.add(accountcodedetailsHashMap);

                for (final AccountPayeeDetail accpayeeDetail : receiptDetail.getAccountPayeeDetails())
                    if (accpayeeDetail.getAmount().compareTo(BigDecimal.ZERO) != 0) {

                        final HashMap<String, Object> subledgerdetailsHashMap = new HashMap<String, Object>();
                        subledgerdetailsHashMap.put(VoucherConstant.GLCODE, accpayeeDetail.getReceiptDetail()
                                .getAccounthead().getGlcode());
                        subledgerdetailsHashMap.put(VoucherConstant.DETAILTYPEID, accpayeeDetail.getAccountDetailType()
                                .getId());
                        subledgerdetailsHashMap.put(VoucherConstant.DETAILKEYID, accpayeeDetail.getAccountDetailKey()
                                .getDetailkey());
                        if (accpayeeDetail.getReceiptDetail().getCramount().compareTo(BigDecimal.ZERO) != 0)
                            subledgerdetailsHashMap.put(VoucherConstant.CREDITAMOUNT, accpayeeDetail.getAmount()
                                    .compareTo(BigDecimal.ZERO) == 0 ? 0 : accpayeeDetail.getAmount());
                        else if (accpayeeDetail.getReceiptDetail().getDramount().compareTo(BigDecimal.ZERO) != 0)
                            subledgerdetailsHashMap.put(VoucherConstant.DEBITAMOUNT, accpayeeDetail.getAmount()
                                    .compareTo(BigDecimal.ZERO) == 0 ? 0 : accpayeeDetail.getAmount());
                        subledgerList.add(subledgerdetailsHashMap);

                    }

            }

        return financialsUtil.createVoucher(headerdetails, accountCodeList, subledgerList, receiptBulkUpload,
                isVoucherApproved);
    }

    /**
     * Creates voucher for given receipt header and maps it with the same.
     *
     * @param receiptHeader
     *            Receipt header for which voucher is to be created
     * @return The created voucher header
     */

    public CVoucherHeader createVoucherForReceipt(final ReceiptHeader receiptHeader, final Boolean receiptBulkUpload)
            throws ApplicationRuntimeException {
        CVoucherHeader voucherheader = null;

        // Additional check for challan Based Receipt, if the receipt cancelled
        // before remittance
        // then need to check the instrument status of that receipt in order to
        // create voucher
        // as the challan has a 'PENDING' receipt object associated with it
        boolean isParentReceiptInstrumentDeposited = false;

        if (receiptHeader.getReceiptHeader() != null)
            for (final InstrumentHeader instrumentHeader : receiptHeader.getReceiptHeader().getReceiptInstrument())
                if (instrumentHeader.getInstrumentType().getType().equals(CollectionConstants.INSTRUMENTTYPE_CASH)) {
                    if (instrumentHeader.getStatusId().getDescription()
                            .equals(CollectionConstants.INSTRUMENT_RECONCILED_STATUS)) {
                        isParentReceiptInstrumentDeposited = true;
                        break;
                    }
                } else if (instrumentHeader.getStatusId().getDescription()
                        .equals(CollectionConstants.INSTRUMENT_DEPOSITED_STATUS)) {
                    isParentReceiptInstrumentDeposited = true;
                    break;
                }

        if (receiptHeader.getReceiptHeader() == null || receiptHeader.getReceiptHeader() != null
                && !isParentReceiptInstrumentDeposited) {
            voucherheader = createVoucher(receiptHeader, receiptBulkUpload);
            if (voucherheader != null) {
                final ReceiptVoucher receiptVoucher = new ReceiptVoucher();
                receiptVoucher.setVoucherheader(voucherheader);
                receiptVoucher.setReceiptHeader(receiptHeader);
                receiptHeader.addReceiptVoucher(receiptVoucher);
            }
        }

        LOGGER.debug("Created voucher for receipt : " + receiptHeader.getReceiptnumber());

        return voucherheader;
    }

    /**
     * Creates vouchers for given set of receipt headers
     *
     * @param receiptHeaders
     *            receipt headers for which vouchers are to be created
     */

    public void createVouchers(final ReceiptHeader receiptHeader, final Boolean receiptBulkUpload)
            throws ApplicationRuntimeException {
        createVoucherForReceipt(receiptHeader, receiptBulkUpload);
    }

    /**
     * Starts workflow for given set of receipt headers. Internally performs the
     * following: 1. Start workflow 2. Transition workflow state with action
     * "Create Receipt" 3. Create vouchers (if required) 4. If vouchers created,
     * transition workflow state with action "Create Voucher"
     *
     * @param receiptHeaders
     *            set of receipt headers on which workflow is to be started
     * @param receiptBulkUpload
     */
    public void startWorkflow(final ReceiptHeader receiptHeader, final Boolean receiptBulkUpload)
            throws ApplicationRuntimeException {
        // Boolean createVoucherForBillingService = Boolean.TRUE;
        // createVoucherForBillingService =
        // receiptHeader.getService().getVoucherCreation() ? Boolean.FALSE :
        // receiptHeader.getService().getVoucherCreation();

        if (receiptHeader.getState() == null) {
            Position position = null;
            if (!collectionsUtil.isEmployee(receiptHeader.getCreatedBy())) {
                final Department dept = departmentService.getDepartmentByName(collectionsUtil
                        .getDepartmentForWorkFlow());
                final Designation desgn = designationService.getDesignationByName(collectionsUtil
                        .getDesignationForThirdPartyUser());
                position = collectionsUtil.getPositionByDeptDesgAndBoundary(dept, desgn, receiptHeader.getReceiptMisc()
                        .getBoundary());
            } else
                position = collectionsUtil.getPositionOfUser(receiptHeader.getCreatedBy());
            receiptHeader.transition().start().withSenderName(receiptHeader.getCreatedBy().getName())
            .withComments(CollectionConstants.WF_STATE_RECEIPT_CREATED)
            .withStateValue(CollectionConstants.WF_STATE_RECEIPT_CREATED).withOwner(position)
            .withDateInfo(new Date()).withNextAction(CollectionConstants.WF_ACTION_SUBMIT);
        }

        LOGGER.debug("Workflow state transition complete");

        /*
         * if (createVoucherForBillingService) { //createVouchers(receiptHeader,
         * receiptBulkUpload); transition(receiptHeader,
         * CollectionConstants.WF_ACTION_CREATE_VOUCHER,
         * "Receipt voucher created"); }
         */
        if (receiptBulkUpload) {
            // transition the receipt header workflow to Approved state
            receiptHeader.transition().withSenderName(receiptHeader.getCreatedBy().getName())
            .withComments("Approval of Data Migration Receipt Complete")
            .withStateValue(CollectionConstants.WF_ACTION_APPROVE)
            .withOwner(collectionsUtil.getPositionOfUser(receiptHeader.getCreatedBy()))
            .withDateInfo(new Date());
            // End the Receipt header workflow
            receiptHeader.transition().end().withSenderName(receiptHeader.getCreatedBy().getName())
            .withComments("Data Migration Receipt Approved - Workflow ends")
            .withStateValue(CollectionConstants.WF_STATE_END)
            .withOwner(collectionsUtil.getPositionOfUser(receiptHeader.getCreatedBy()))
            .withDateInfo(new Date());
        }
    }

    /**
     * Method to find all the Cash,Cheque and DD type instruments with status as
     * :new and
     *
     * @return List of HashMap
     */
    public List<HashMap<String, Object>> findAllRemitanceDetails(final String boundaryIdList) {
        final List<HashMap<String, Object>> paramList = new ArrayList<HashMap<String, Object>>();
        // TODO: Fix the sum(ih.instrumentamount) the amount is wrong because of
        // the ujl.boundary in (" + boundaryIdList + ")"
        final String queryBuilder = "SELECT sum(ih.instrumentamount) as INSTRUMENTMAOUNT,to_char(ch.RECEIPTDATE, 'DD-MM-YYYY') AS RECEIPTDATE,"
                + "sd.NAME as SERVICENAME,it.TYPE as INSTRUMENTTYPE,fnd.name AS FUNDNAME,dpt.name AS DEPARTMENTNAME,"
                + "fnd.code AS FUNDCODE,dpt.code AS DEPARTMENTCODE from EGCL_COLLECTIONHEADER ch,"
                + "EGF_INSTRUMENTHEADER ih,EGCL_COLLECTIONINSTRUMENT ci,EGCL_SERVICEDETAILS sd,"
                + "EGF_INSTRUMENTTYPE it,EGCL_COLLECTIONMIS cm,FUND fnd,EG_DEPARTMENT dpt";

        final String whereClauseBeforInstumentType = " where ch.id=cm.collectionheader AND "
                + "fnd.id=cm.fund AND dpt.id=cm.department and ci.INSTRUMENTHEADER=ih.ID and "
                + "ch.SERVICEDETAILS=sd.ID and ch.ID=ci.COLLECTIONHEADER and ih.INSTRUMENTTYPE=it.ID and ";

        final String whereClause = " AND ih.ID_STATUS=(select id from egw_status where moduletype='"
                + CollectionConstants.MODULE_NAME_INSTRUMENTHEADER + "' " + "and description='"
                + CollectionConstants.INSTRUMENT_NEW_STATUS
                + "') and ih.ISPAYCHEQUE='0' and ch.STATUS=(select id from egw_status where " + "moduletype='"
                + CollectionConstants.MODULE_NAME_RECEIPTHEADER + "' and code='"
                + CollectionConstants.RECEIPT_STATUS_CODE_APPROVED + "') ";

        final String groupByClause = " group by to_char(ch.RECEIPTDATE, 'DD-MM-YYYY'),sd.NAME,it.TYPE,fnd.name,dpt.name,fnd.code,dpt.code";
        final String orderBy = " order by RECEIPTDATE";

        /**
         * Query to get the collection of the instrument types Cash,Cheque,DD &
         * Card for bank remittance
         */
        final StringBuilder queryStringForCashChequeDDCard = new StringBuilder(queryBuilder + ",egeis_jurisdiction ujl"
                + whereClauseBeforInstumentType + "it.TYPE in ('" + CollectionConstants.INSTRUMENTTYPE_CASH + "','"
                + CollectionConstants.INSTRUMENTTYPE_CHEQUE + "'," + "'" + CollectionConstants.INSTRUMENTTYPE_DD
                + "','" + CollectionConstants.INSTRUMENTTYPE_CARD + "') " + whereClause
                + "AND ch.CREATEDBY=ujl.employee and ujl.boundary in (" + boundaryIdList + ")" + groupByClause);

        /**
         * If the department of login user is AccountCell .i.e., Department
         * Code-'A',then this user will be able to remit online transaction as
         * well. All the online receipts created by 'citizen' user will be
         * remitted by Account Cell user.
         */
        final User citizenUser = collectionsUtil.getUserByUserName(CollectionConstants.CITIZEN_USER_NAME);

        if (boundaryIdList != null && citizenUser != null) {
            final String queryStringForOnline = " union " + queryBuilder + whereClauseBeforInstumentType + "it.TYPE='"
                    + CollectionConstants.INSTRUMENTTYPE_ONLINE + "'" + whereClause + "AND ch.CREATEDBY="
                    + citizenUser.getId() + groupByClause;

            queryStringForCashChequeDDCard.append(queryStringForOnline);
        }

        final Query query = getSession().createSQLQuery(queryStringForCashChequeDDCard.toString() + orderBy);

        final List<Object[]> queryResults = query.list();

        for (int i = 0; i < queryResults.size(); i++) {
            final Object[] arrayObjectInitialIndex = queryResults.get(i);
            HashMap<String, Object> objHashMap = new HashMap<String, Object>();

            if (i == 0) {
                objHashMap.put(CollectionConstants.BANKREMITTANCE_RECEIPTDATE, arrayObjectInitialIndex[1]);
                objHashMap.put(CollectionConstants.BANKREMITTANCE_SERVICENAME, arrayObjectInitialIndex[2]);
                objHashMap.put(CollectionConstants.BANKREMITTANCE_FUNDNAME, arrayObjectInitialIndex[4]);
                objHashMap.put(CollectionConstants.BANKREMITTANCE_DEPARTMENTNAME, arrayObjectInitialIndex[5]);
                objHashMap.put(CollectionConstants.BANKREMITTANCE_FUNDCODE, arrayObjectInitialIndex[6]);
                objHashMap.put(CollectionConstants.BANKREMITTANCE_DEPARTMENTCODE, arrayObjectInitialIndex[7]);

                if (arrayObjectInitialIndex[3].equals(CollectionConstants.INSTRUMENTTYPE_CASH)) {
                    objHashMap.put(CollectionConstants.BANKREMITTANCE_SERVICETOTALCASHAMOUNT,
                            arrayObjectInitialIndex[0]);
                    objHashMap.put(CollectionConstants.BANKREMITTANCE_SERVICETOTALCHEQUEAMOUNT, "");
                    objHashMap.put(CollectionConstants.BANKREMITTANCE_SERVICETOTALCARDPAYMENTAMOUNT, "");
                    objHashMap.put(CollectionConstants.BANKREMITTANCE_SERVICETOTALONLINEPAYMENTAMOUNT, "");
                }
                if (arrayObjectInitialIndex[3].equals(CollectionConstants.INSTRUMENTTYPE_CHEQUE)
                        || arrayObjectInitialIndex[3].equals(CollectionConstants.INSTRUMENTTYPE_DD)) {
                    objHashMap.put(CollectionConstants.BANKREMITTANCE_SERVICETOTALCASHAMOUNT, "");
                    objHashMap.put(CollectionConstants.BANKREMITTANCE_SERVICETOTALCHEQUEAMOUNT,
                            arrayObjectInitialIndex[0]);
                    objHashMap.put(CollectionConstants.BANKREMITTANCE_SERVICETOTALCARDPAYMENTAMOUNT, "");
                    objHashMap.put(CollectionConstants.BANKREMITTANCE_SERVICETOTALONLINEPAYMENTAMOUNT, "");
                }
                if (arrayObjectInitialIndex[3].equals(CollectionConstants.INSTRUMENTTYPE_CARD)) {
                    objHashMap.put(CollectionConstants.BANKREMITTANCE_SERVICETOTALCASHAMOUNT, "");
                    objHashMap.put(CollectionConstants.BANKREMITTANCE_SERVICETOTALCHEQUEAMOUNT, "");
                    objHashMap.put(CollectionConstants.BANKREMITTANCE_SERVICETOTALCARDPAYMENTAMOUNT,
                            arrayObjectInitialIndex[0]);
                    objHashMap.put(CollectionConstants.BANKREMITTANCE_SERVICETOTALONLINEPAYMENTAMOUNT, "");
                }
                if (arrayObjectInitialIndex[3].equals(CollectionConstants.INSTRUMENTTYPE_ONLINE)) {
                    objHashMap.put(CollectionConstants.BANKREMITTANCE_SERVICETOTALCASHAMOUNT, "");
                    objHashMap.put(CollectionConstants.BANKREMITTANCE_SERVICETOTALCHEQUEAMOUNT, "");
                    objHashMap.put(CollectionConstants.BANKREMITTANCE_SERVICETOTALCARDPAYMENTAMOUNT, "");
                    objHashMap.put(CollectionConstants.BANKREMITTANCE_SERVICETOTALONLINEPAYMENTAMOUNT,
                            arrayObjectInitialIndex[0]);
                }
            } else {
                final int checknew = checkIfMapObjectExist(paramList, arrayObjectInitialIndex);
                if (checknew == -1) {
                    objHashMap.put(CollectionConstants.BANKREMITTANCE_RECEIPTDATE, arrayObjectInitialIndex[1]);
                    objHashMap.put(CollectionConstants.BANKREMITTANCE_SERVICENAME, arrayObjectInitialIndex[2]);
                    objHashMap.put(CollectionConstants.BANKREMITTANCE_FUNDNAME, arrayObjectInitialIndex[4]);
                    objHashMap.put(CollectionConstants.BANKREMITTANCE_DEPARTMENTNAME, arrayObjectInitialIndex[5]);
                    objHashMap.put(CollectionConstants.BANKREMITTANCE_FUNDCODE, arrayObjectInitialIndex[6]);
                    objHashMap.put(CollectionConstants.BANKREMITTANCE_DEPARTMENTCODE, arrayObjectInitialIndex[7]);

                    if (arrayObjectInitialIndex[3].equals(CollectionConstants.INSTRUMENTTYPE_CASH)) {
                        objHashMap.put(CollectionConstants.BANKREMITTANCE_SERVICETOTALCASHAMOUNT,
                                arrayObjectInitialIndex[0]);
                        objHashMap.put(CollectionConstants.BANKREMITTANCE_SERVICETOTALCHEQUEAMOUNT, "");
                        objHashMap.put(CollectionConstants.BANKREMITTANCE_SERVICETOTALCARDPAYMENTAMOUNT, "");
                        objHashMap.put(CollectionConstants.BANKREMITTANCE_SERVICETOTALONLINEPAYMENTAMOUNT, "");
                    }
                    if (arrayObjectInitialIndex[3].equals(CollectionConstants.INSTRUMENTTYPE_CHEQUE)
                            || arrayObjectInitialIndex[3].equals(CollectionConstants.INSTRUMENTTYPE_DD)) {
                        objHashMap.put(CollectionConstants.BANKREMITTANCE_SERVICETOTALCASHAMOUNT, "");
                        objHashMap.put(CollectionConstants.BANKREMITTANCE_SERVICETOTALCHEQUEAMOUNT,
                                arrayObjectInitialIndex[0]);
                        objHashMap.put(CollectionConstants.BANKREMITTANCE_SERVICETOTALCARDPAYMENTAMOUNT, "");
                        objHashMap.put(CollectionConstants.BANKREMITTANCE_SERVICETOTALONLINEPAYMENTAMOUNT, "");
                    }
                    if (arrayObjectInitialIndex[3].equals(CollectionConstants.INSTRUMENTTYPE_CARD)) {
                        objHashMap.put(CollectionConstants.BANKREMITTANCE_SERVICETOTALCASHAMOUNT, "");
                        objHashMap.put(CollectionConstants.BANKREMITTANCE_SERVICETOTALCHEQUEAMOUNT, "");
                        objHashMap.put(CollectionConstants.BANKREMITTANCE_SERVICETOTALCARDPAYMENTAMOUNT,
                                arrayObjectInitialIndex[0]);
                        objHashMap.put(CollectionConstants.BANKREMITTANCE_SERVICETOTALONLINEPAYMENTAMOUNT, "");
                    }
                    if (arrayObjectInitialIndex[3].equals(CollectionConstants.INSTRUMENTTYPE_ONLINE)) {
                        objHashMap.put(CollectionConstants.BANKREMITTANCE_SERVICETOTALCASHAMOUNT, "");
                        objHashMap.put(CollectionConstants.BANKREMITTANCE_SERVICETOTALCHEQUEAMOUNT, "");
                        objHashMap.put(CollectionConstants.BANKREMITTANCE_SERVICETOTALCARDPAYMENTAMOUNT, "");
                        objHashMap.put(CollectionConstants.BANKREMITTANCE_SERVICETOTALONLINEPAYMENTAMOUNT,
                                arrayObjectInitialIndex[0]);
                    }
                } else {
                    objHashMap = paramList.get(checknew);

                    paramList.remove(checknew);

                    if (arrayObjectInitialIndex[3].equals(CollectionConstants.INSTRUMENTTYPE_CASH))
                        objHashMap.put(CollectionConstants.BANKREMITTANCE_SERVICETOTALCASHAMOUNT,
                                arrayObjectInitialIndex[0]);
                    if (arrayObjectInitialIndex[3].equals(CollectionConstants.INSTRUMENTTYPE_CHEQUE)
                            || arrayObjectInitialIndex[3].equals(CollectionConstants.INSTRUMENTTYPE_DD)) {
                        BigDecimal existingAmount = BigDecimal.ZERO;
                        if (objHashMap.get(CollectionConstants.BANKREMITTANCE_SERVICETOTALCHEQUEAMOUNT) != "")
                            existingAmount = new BigDecimal(objHashMap.get(
                                    CollectionConstants.BANKREMITTANCE_SERVICETOTALCHEQUEAMOUNT).toString());
                        existingAmount = existingAmount.add(new BigDecimal(arrayObjectInitialIndex[0].toString()));
                        objHashMap.put(CollectionConstants.BANKREMITTANCE_SERVICETOTALCHEQUEAMOUNT, existingAmount);
                    }
                    if (arrayObjectInitialIndex[3].equals(CollectionConstants.INSTRUMENTTYPE_CARD))
                        objHashMap.put(CollectionConstants.BANKREMITTANCE_SERVICETOTALCARDPAYMENTAMOUNT,
                                arrayObjectInitialIndex[0]);
                    if (arrayObjectInitialIndex[3].equals(CollectionConstants.INSTRUMENTTYPE_ONLINE))
                        objHashMap.put(CollectionConstants.BANKREMITTANCE_SERVICETOTALONLINEPAYMENTAMOUNT,
                                arrayObjectInitialIndex[0]);
                }
            }
            if (objHashMap.get(CollectionConstants.BANKREMITTANCE_RECEIPTDATE) != null
                    && objHashMap.get(CollectionConstants.BANKREMITTANCE_SERVICENAME) != null)
                paramList.add(objHashMap);
        }
        return paramList;
    }

    /**
     * Method to check if the given HashMap already exists in the List of
     * HashMap
     *
     * @param queryResults
     * @param objHashMap
     * @param m
     * @return index of objHashMap in the queryResults
     */
    public int checkIfMapObjectExist(final List<HashMap<String, Object>> paramList,
            final Object[] arrayObjectInitialIndexTemp) {
        int check = -1;
        for (int m = 0; m < paramList.size(); m++) {
            final HashMap<String, Object> objHashMapTemp = paramList.get(m);

            if (arrayObjectInitialIndexTemp[1] != null && arrayObjectInitialIndexTemp[2] != null)
                if (arrayObjectInitialIndexTemp[1].equals(objHashMapTemp
                        .get(CollectionConstants.BANKREMITTANCE_RECEIPTDATE))
                        && arrayObjectInitialIndexTemp[2].equals(objHashMapTemp
                                .get(CollectionConstants.BANKREMITTANCE_SERVICENAME))
                                && arrayObjectInitialIndexTemp[6].equals(objHashMapTemp
                                        .get(CollectionConstants.BANKREMITTANCE_FUNDCODE))
                                        && arrayObjectInitialIndexTemp[7].equals(objHashMapTemp
                                                .get(CollectionConstants.BANKREMITTANCE_DEPARTMENTCODE))) {
                    check = m;
                    break;
                } else
                    continue;

        }
        return check;
    }

    /**
     * Create Contra Vouchers for String array passed of serviceName,
     * totalCashAmount, totalChequeAmount, totalCardAmount and totalOnlineAcount
     *
     * @param serviceName
     * @param totalCashAmount
     * @param totalChequeAmount
     * @return List of Contra Vouchers Created
     */
    public List createBankRemittance(final String[] serviceNameArr, final String[] totalCashAmount,
            final String[] totalChequeAmount, final String[] totalCardAmount, final String[] totalOnlineAmount,
            final String[] receiptDateArray, final String[] fundCodeArray, final String[] departmentCodeArray,
            final Integer accountNumberId, final Integer positionUser, final String[] receiptNumberArray) {
        final List<CVoucherHeader> newContraVoucherList = new ArrayList<CVoucherHeader>();
        final List<ReceiptHeader> bankRemittanceList = new ArrayList<ReceiptHeader>();
        final SimpleDateFormat dateFomatter = new SimpleDateFormat("yyyy-MM-dd", Locale.getDefault());
        final Map<String, Object> instrumentDepositeMap = financialsUtil.prepareForUpdateInstrumentDepositSQL();
        final String instrumentGlCodeQueryString = "SELECT COA.GLCODE FROM CHARTOFACCOUNTS COA,EGF_INSTRUMENTACCOUNTCODES IAC,EGF_INSTRUMENTTYPE IT "
                + "WHERE IT.ID=IAC.TYPEID AND IAC.GLCODEID=COA.ID AND IT.TYPE=";
        final String receiptInstrumentQueryString = "select DISTINCT (instruments) from org.egov.collection.entity.ReceiptHeader receipt "
                + "join receipt.receiptInstrument as instruments where ";
        final String serviceNameCondition = "receipt.service.name=? ";
        final String receiptDateCondition = "and to_char(receipt.createdDate,'dd-MM-yyyy')=? ";
        final String instrumentStatusCondition = "and instruments.statusId.id=? ";
        final String instrumentTypeCondition = "and instruments.instrumentType.type = ? ";
        final String receiptFundCondition = "and receipt.receiptMisc.fund.code = ? ";
        final String receiptDepartmentCondition = "and receipt.receiptMisc.department.code = ? ";

        final String cashInHandQueryString = instrumentGlCodeQueryString + "'"
                + CollectionConstants.INSTRUMENTTYPE_CASH + "'";
        final String chequeInHandQueryString = instrumentGlCodeQueryString + "'"
                + CollectionConstants.INSTRUMENTTYPE_CHEQUE + "'";
        final String cardPaymentQueryString = instrumentGlCodeQueryString + "'"
                + CollectionConstants.INSTRUMENTTYPE_CARD + "'";
        final String onlinePaymentQueryString = instrumentGlCodeQueryString + "'"
                + CollectionConstants.INSTRUMENTTYPE_ONLINE + "'";

        final Query cashInHand = getSession().createSQLQuery(cashInHandQueryString);
        final Query chequeInHand = getSession().createSQLQuery(chequeInHandQueryString);
        final Query cardPaymentAccount = getSession().createSQLQuery(cardPaymentQueryString);
        final Query onlinePaymentAccount = getSession().createSQLQuery(onlinePaymentQueryString);

        String cashInHandGLCode = null, chequeInHandGlcode = null, cardPaymentGlCode = null, onlinePaymentGlCode = null;

        final String voucherWorkflowMsg = "Voucher Workflow Started";

        if (!cashInHand.list().isEmpty())
            cashInHandGLCode = cashInHand.list().get(0).toString();
        if (!chequeInHand.list().isEmpty())
            chequeInHandGlcode = chequeInHand.list().get(0).toString();
        if (!cardPaymentAccount.list().isEmpty())
            cardPaymentGlCode = cardPaymentAccount.list().get(0).toString();
        if (!onlinePaymentAccount.list().isEmpty())
            onlinePaymentGlCode = onlinePaymentAccount.list().get(0).toString();

        final EgwStatus status = egwStatusDAO.getStatusByModuleAndCode(
                CollectionConstants.MODULE_NAME_INSTRUMENTHEADER, CollectionConstants.INSTRUMENT_NEW_STATUS);

        /**
         * Get the AppConfig parameter defined for the Remittance voucher type
         * in case of instrument type Cheque,DD & Card
         */

        Boolean voucherTypeForChequeDDCard = false;
        Boolean useReceiptDateAsContraVoucherDate = false;

        if (collectionsUtil.getAppConfigValue(CollectionConstants.MODULE_NAME_COLLECTIONS_CONFIG,
                CollectionConstants.APPCONFIG_VALUE_REMITTANCEVOUCHERTYPEFORCHEQUEDDCARD).equals(
                        CollectionConstants.FINANCIAL_RECEIPTS_VOUCHERTYPE))
            voucherTypeForChequeDDCard = true;

        if (collectionsUtil.getAppConfigValue(CollectionConstants.MODULE_NAME_COLLECTIONS_CONFIG,
                CollectionConstants.APPCONFIG_VALUE_USERECEIPTDATEFORCONTRA).equals(CollectionConstants.YES))
            useReceiptDateAsContraVoucherDate = true;

        for (int i = 0; i < serviceNameArr.length; i++) {
            final String serviceName = serviceNameArr[i].trim();
            Date voucherDate = new Date();

            if (useReceiptDateAsContraVoucherDate)
                try {
                    voucherDate = dateFomatter.parse(receiptDateArray[i]);
                } catch (final ParseException exp) {
                    LOGGER.debug("Exception in parsing date  " + receiptDateArray[i] + " - " + exp.getMessage());
                }

            if (serviceName != null && serviceName.length() > 0) {
                final String serviceGLCodeQueryString = "select coa.glcode from BANKACCOUNT ba,CHARTOFACCOUNTS coa where "
                        + "ba.GLCODEID=coa.ID and ba.ID=" + accountNumberId;
                persistenceService.findByNamedQuery(CollectionConstants.QUERY_SERVICE_BY_NAME, serviceName);
                Bankaccount depositedBankAccount = null;
                final ServiceDetails serviceDetails = (ServiceDetails) persistenceService.findByNamedQuery(
                        CollectionConstants.QUERY_SERVICE_BY_NAME, serviceName);
                final Query serviceGLCodeQuery = getSession().createSQLQuery(serviceGLCodeQueryString);

                final String serviceGlCode = serviceGLCodeQuery.list().get(0).toString();
                final List<HashMap<String, Object>> subledgerList = new ArrayList<HashMap<String, Object>>();

                // If Cash Amount is present
                if (totalCashAmount[i].trim() != null && totalCashAmount[i].trim().length() > 0
                        && cashInHandGLCode != null) {
                    final StringBuilder cashQueryBuilder = new StringBuilder(receiptInstrumentQueryString);
                    cashQueryBuilder.append(serviceNameCondition);
                    cashQueryBuilder.append(receiptDateCondition);
                    cashQueryBuilder.append(instrumentStatusCondition);
                    cashQueryBuilder.append(instrumentTypeCondition);
                    cashQueryBuilder.append(receiptFundCondition);
                    cashQueryBuilder.append(receiptDepartmentCondition);

                    final Object arguments[] = new Object[6];

                    arguments[0] = serviceName;
                    arguments[1] = receiptDateArray[i];
                    arguments[2] = status.getId();
                    arguments[3] = CollectionConstants.INSTRUMENTTYPE_CASH;
                    arguments[4] = fundCodeArray[i];
                    arguments[5] = departmentCodeArray[i];

                    final List<InstrumentHeader> instrumentHeaderListCash = persistenceService.findAllBy(
                            cashQueryBuilder.toString(), arguments);

                    final HashMap<String, Object> headerdetails = new HashMap<String, Object>();

                    headerdetails.put(VoucherConstant.VOUCHERNAME,
                            CollectionConstants.FINANCIAL_CONTRATVOUCHER_VOUCHERNAME);
                    headerdetails.put(VoucherConstant.VOUCHERTYPE,
                            CollectionConstants.FINANCIAL_CONTRAVOUCHER_VOUCHERTYPE);
                    headerdetails.put(VoucherConstant.DESCRIPTION, CollectionConstants.FINANCIAL_VOUCHERDESCRIPTION);
                    headerdetails.put(VoucherConstant.VOUCHERDATE, voucherDate);
                    headerdetails.put(VoucherConstant.FUNDCODE, fundCodeArray[i]);
                    headerdetails.put(VoucherConstant.DEPARTMENTCODE, departmentCodeArray[i]);
                    headerdetails.put(VoucherConstant.FUNDSOURCECODE, serviceDetails.getFundSource() == null ? null
                            : serviceDetails.getFundSource().getCode());
                    headerdetails.put(VoucherConstant.FUNCTIONARYCODE, serviceDetails.getFunctionary() == null ? null
                            : serviceDetails.getFunctionary().getCode());
                    headerdetails.put(VoucherConstant.MODULEID, CollectionConstants.COLLECTIONS_EG_MODULES_ID);
                    headerdetails.put(VoucherConstant.MODULEID, CollectionConstants.COLLECTIONS_EG_MODULES_ID);

                    final List<HashMap<String, Object>> accountCodeCashList = new ArrayList<HashMap<String, Object>>();
                    final HashMap<String, Object> accountcodedetailsCreditCashHashMap = new HashMap<String, Object>();

                    accountcodedetailsCreditCashHashMap.put(VoucherConstant.GLCODE, cashInHandGLCode);
                    accountcodedetailsCreditCashHashMap.put(VoucherConstant.FUNCTIONCODE, null);
                    accountcodedetailsCreditCashHashMap.put(VoucherConstant.CREDITAMOUNT, totalCashAmount[i]);
                    accountcodedetailsCreditCashHashMap.put(VoucherConstant.DEBITAMOUNT, 0);

                    accountCodeCashList.add(accountcodedetailsCreditCashHashMap);
                    // TODO: Add debit account details
                    {
                        final HashMap<String, Object> accountcodedetailsDebitHashMap = new HashMap<String, Object>();
                        accountcodedetailsDebitHashMap.put(VoucherConstant.GLCODE, serviceGlCode);
                        accountcodedetailsDebitHashMap.put(VoucherConstant.FUNCTIONCODE, null);
                        accountcodedetailsDebitHashMap.put(VoucherConstant.CREDITAMOUNT, 0);
                        accountcodedetailsDebitHashMap.put(VoucherConstant.DEBITAMOUNT, totalCashAmount[i]);
                        accountCodeCashList.add(accountcodedetailsDebitHashMap);
                    }

                    final CVoucherHeader voucherHeaderCash = financialsUtil.createRemittanceVoucher(headerdetails,
                            accountCodeCashList, subledgerList);

                    newContraVoucherList.add(voucherHeaderCash);
                    depositedBankAccount = (Bankaccount) persistenceService.find(
                            "from Bankaccount where chartofaccounts.glcode=?", serviceGlCode);
                    if (voucherHeaderCash != null && voucherHeaderCash.getId() != null)
                        createVoucherForCashRemittance(instrumentDepositeMap, voucherWorkflowMsg, voucherDate,
                                depositedBankAccount, serviceGlCode, instrumentHeaderListCash, voucherHeaderCash);
                    else {
                        final EgwStatus statusDeposited = egwStatusDAO.getStatusByModuleAndCode(
                                CollectionConstants.MODULE_NAME_INSTRUMENTHEADER,
                                CollectionConstants.INSTRUMENT_DEPOSITED_STATUS);
                        financialsUtil.updateInstrumentHeader(instrumentHeaderListCash, statusDeposited,
                                depositedBankAccount);
                    }

                    for (final InstrumentHeader instHead : instrumentHeaderListCash) {
                        final List<ReceiptHeader> receiptHeaders = findAllByNamedQuery(
                                CollectionConstants.QUERY_RECEIPTS_BY_INSTRUMENTHEADER_AND_SERVICECODE,
                                instHead.getId(), serviceDetails.getCode());
                        bankRemittanceList.addAll(receiptHeaders);
                    }

                }
                // If Cheque Amount is present
                if (totalChequeAmount[i].trim() != null && totalChequeAmount[i].trim().length() > 0
                        && chequeInHandGlcode != null) {
                    final StringBuilder chequeQueryBuilder = new StringBuilder(receiptInstrumentQueryString);
                    chequeQueryBuilder.append(serviceNameCondition);
                    chequeQueryBuilder.append(receiptDateCondition);
                    chequeQueryBuilder.append(instrumentStatusCondition);
                    chequeQueryBuilder.append("and instruments.instrumentType.type in ( ?, ?)");
                    chequeQueryBuilder
                    .append("and receipt.status.id=(select id from org.egov.commons.EgwStatus where moduletype=? and code=?) ");
                    chequeQueryBuilder.append(receiptFundCondition);
                    chequeQueryBuilder.append(receiptDepartmentCondition);

                    final Object arguments[] = new Object[9];

                    arguments[0] = serviceName;
                    arguments[1] = receiptDateArray[i];
                    arguments[2] = status.getId();
                    arguments[3] = CollectionConstants.INSTRUMENTTYPE_CHEQUE;
                    arguments[4] = CollectionConstants.INSTRUMENTTYPE_DD;
                    arguments[5] = CollectionConstants.MODULE_NAME_RECEIPTHEADER;
                    arguments[6] = CollectionConstants.RECEIPT_STATUS_CODE_APPROVED;
                    arguments[7] = fundCodeArray[i];
                    arguments[8] = departmentCodeArray[i];

                    final List<InstrumentHeader> instrumentHeaderListCheque = persistenceService.findAllBy(
                            chequeQueryBuilder.toString(), arguments);
                    final HashMap<String, Object> headerdetails = new HashMap<String, Object>();
                    final List<HashMap<String, Object>> accountCodeChequeList = new ArrayList<HashMap<String, Object>>();
                    final HashMap<String, Object> accountcodedetailsCreditChequeHashMap = new HashMap<String, Object>();

                    if (voucherTypeForChequeDDCard) {
                        headerdetails.put(VoucherConstant.VOUCHERNAME,
                                CollectionConstants.FINANCIAL_RECEIPTS_VOUCHERNAME);
                        headerdetails.put(VoucherConstant.VOUCHERTYPE,
                                CollectionConstants.FINANCIAL_RECEIPTS_VOUCHERTYPE);

                    } else {
                        headerdetails.put(VoucherConstant.VOUCHERNAME,
                                CollectionConstants.FINANCIAL_CONTRATVOUCHER_VOUCHERNAME);
                        headerdetails.put(VoucherConstant.VOUCHERTYPE,
                                CollectionConstants.FINANCIAL_CONTRAVOUCHER_VOUCHERTYPE);
                    }
                    headerdetails.put(VoucherConstant.VOUCHERNAME,
                            CollectionConstants.FINANCIAL_CONTRATVOUCHER_VOUCHERNAME);
                    headerdetails.put(VoucherConstant.VOUCHERTYPE,
                            CollectionConstants.FINANCIAL_CONTRAVOUCHER_VOUCHERTYPE);
                    headerdetails.put(VoucherConstant.DESCRIPTION, CollectionConstants.FINANCIAL_VOUCHERDESCRIPTION);
                    headerdetails.put(VoucherConstant.VOUCHERDATE, voucherDate);
                    headerdetails.put(VoucherConstant.FUNDCODE, fundCodeArray[i]);
                    headerdetails.put(VoucherConstant.DEPARTMENTCODE, departmentCodeArray[i]);
                    headerdetails.put(VoucherConstant.FUNDSOURCECODE, serviceDetails.getFundSource() == null ? null
                            : serviceDetails.getFundSource().getCode());
                    headerdetails.put(VoucherConstant.FUNCTIONARYCODE, serviceDetails.getFunctionary() == null ? null
                            : serviceDetails.getFunctionary().getCode());
                    headerdetails.put(VoucherConstant.MODULEID, CollectionConstants.COLLECTIONS_EG_MODULES_ID);

                    accountcodedetailsCreditChequeHashMap.put(VoucherConstant.GLCODE, chequeInHandGlcode);
                    accountcodedetailsCreditChequeHashMap.put(VoucherConstant.FUNCTIONCODE, null);
                    accountcodedetailsCreditChequeHashMap.put(VoucherConstant.CREDITAMOUNT, totalChequeAmount[i]);
                    accountcodedetailsCreditChequeHashMap.put(VoucherConstant.DEBITAMOUNT, 0);

                    accountCodeChequeList.add(accountcodedetailsCreditChequeHashMap);
                    // TODO: Add debit account details
                    {
                        final HashMap<String, Object> accountcodedetailsDebitHashMap = new HashMap<String, Object>();
                        accountcodedetailsDebitHashMap.put(VoucherConstant.GLCODE, serviceGlCode);
                        accountcodedetailsDebitHashMap.put(VoucherConstant.FUNCTIONCODE, null);
                        accountcodedetailsDebitHashMap.put(VoucherConstant.CREDITAMOUNT, 0);
                        accountcodedetailsDebitHashMap.put(VoucherConstant.DEBITAMOUNT, totalChequeAmount[i]);
                        accountCodeChequeList.add(accountcodedetailsDebitHashMap);
                    }
                    final CVoucherHeader voucherHeaderCheque = financialsUtil.createRemittanceVoucher(headerdetails,
                            accountCodeChequeList, subledgerList);

                    newContraVoucherList.add(voucherHeaderCheque);
                    depositedBankAccount = (Bankaccount) persistenceService.find(
                            "from Bankaccount where chartofaccounts.glcode=?", serviceGlCode);
                    if (voucherHeaderCheque != null && voucherHeaderCheque.getId() != null)
                        createVoucherForChequeCardRemittance(instrumentDepositeMap, voucherWorkflowMsg,
                                voucherTypeForChequeDDCard, voucherDate, depositedBankAccount, serviceGlCode,
                                instrumentHeaderListCheque, voucherHeaderCheque);
                    else {
                        final EgwStatus statusDeposited = egwStatusDAO.getStatusByModuleAndCode(
                                CollectionConstants.MODULE_NAME_INSTRUMENTHEADER,
                                CollectionConstants.INSTRUMENT_DEPOSITED_STATUS);
                        financialsUtil.updateInstrumentHeader(instrumentHeaderListCheque, statusDeposited,
                                depositedBankAccount);
                    }

                    for (final InstrumentHeader instHead : instrumentHeaderListCheque) {
                        final List<ReceiptHeader> receiptHeaders = findAllByNamedQuery(
                                CollectionConstants.QUERY_RECEIPTS_BY_INSTRUMENTHEADER_AND_SERVICECODE,
                                instHead.getId(), serviceDetails.getCode());
                        bankRemittanceList.addAll(receiptHeaders);
                    }
                }
                // If card amount is present
                if (totalCardAmount[i].trim() != null && totalCardAmount[i].trim().length() > 0
                        && cardPaymentGlCode != null) {
                    final StringBuilder onlineQueryBuilder = new StringBuilder(receiptInstrumentQueryString);
                    onlineQueryBuilder.append(serviceNameCondition);
                    onlineQueryBuilder.append(receiptDateCondition);
                    onlineQueryBuilder.append(instrumentStatusCondition);
                    onlineQueryBuilder.append(instrumentTypeCondition);
                    onlineQueryBuilder.append(receiptFundCondition);
                    onlineQueryBuilder.append(receiptDepartmentCondition);

                    final Object arguments[] = new Object[6];

                    arguments[0] = serviceName;
                    arguments[1] = receiptDateArray[i];
                    arguments[2] = status.getId();
                    arguments[3] = CollectionConstants.INSTRUMENTTYPE_CARD;
                    arguments[4] = fundCodeArray[i];
                    arguments[5] = departmentCodeArray[i];

                    final List<InstrumentHeader> instrumentHeaderListOnline = persistenceService.findAllBy(
                            onlineQueryBuilder.toString(), arguments);

                    final HashMap<String, Object> headerdetails = new HashMap<String, Object>();

                    if (voucherTypeForChequeDDCard) {
                        headerdetails.put(VoucherConstant.VOUCHERNAME,
                                CollectionConstants.FINANCIAL_RECEIPTS_VOUCHERNAME);
                        headerdetails.put(VoucherConstant.VOUCHERTYPE,
                                CollectionConstants.FINANCIAL_RECEIPTS_VOUCHERTYPE);
                    } else {

                        headerdetails.put(VoucherConstant.VOUCHERNAME,
                                CollectionConstants.FINANCIAL_CONTRATVOUCHER_VOUCHERNAME);
                        headerdetails.put(VoucherConstant.VOUCHERTYPE,
                                CollectionConstants.FINANCIAL_CONTRAVOUCHER_VOUCHERTYPE);
                    }
                    headerdetails.put(VoucherConstant.VOUCHERNAME,
                            CollectionConstants.FINANCIAL_CONTRATVOUCHER_VOUCHERNAME);
                    headerdetails.put(VoucherConstant.VOUCHERTYPE,
                            CollectionConstants.FINANCIAL_CONTRAVOUCHER_VOUCHERTYPE);
                    headerdetails.put(VoucherConstant.DESCRIPTION, CollectionConstants.FINANCIAL_VOUCHERDESCRIPTION);
                    headerdetails.put(VoucherConstant.VOUCHERDATE, voucherDate);
                    headerdetails.put(VoucherConstant.FUNDCODE, fundCodeArray[i]);
                    headerdetails.put(VoucherConstant.DEPARTMENTCODE, departmentCodeArray[i]);
                    headerdetails.put(VoucherConstant.FUNDSOURCECODE, serviceDetails.getFundSource() == null ? null
                            : serviceDetails.getFundSource().getCode());
                    headerdetails.put(VoucherConstant.FUNCTIONARYCODE, serviceDetails.getFunctionary() == null ? null
                            : serviceDetails.getFunctionary().getCode());
                    headerdetails.put(VoucherConstant.MODULEID, CollectionConstants.COLLECTIONS_EG_MODULES_ID);

                    final List<HashMap<String, Object>> accountCodeOnlineList = new ArrayList<HashMap<String, Object>>();
                    final HashMap<String, Object> accountcodedetailsCreditOnlineHashMap = new HashMap<String, Object>();

                    accountcodedetailsCreditOnlineHashMap.put(VoucherConstant.GLCODE, cardPaymentGlCode);
                    accountcodedetailsCreditOnlineHashMap.put(VoucherConstant.FUNCTIONCODE, null);
                    accountcodedetailsCreditOnlineHashMap.put(VoucherConstant.CREDITAMOUNT, totalCardAmount[i]);
                    accountcodedetailsCreditOnlineHashMap.put(VoucherConstant.DEBITAMOUNT, 0);

                    accountCodeOnlineList.add(accountcodedetailsCreditOnlineHashMap);
                    // TODO: Add debit account details
                    {
                        final HashMap<String, Object> accountcodedetailsDebitHashMap = new HashMap<String, Object>();
                        accountcodedetailsDebitHashMap.put(VoucherConstant.GLCODE, serviceGlCode);
                        accountcodedetailsDebitHashMap.put(VoucherConstant.FUNCTIONCODE, null);
                        accountcodedetailsDebitHashMap.put(VoucherConstant.CREDITAMOUNT, 0);
                        accountcodedetailsDebitHashMap.put(VoucherConstant.DEBITAMOUNT, totalCardAmount[i]);
                        accountCodeOnlineList.add(accountcodedetailsDebitHashMap);
                    }
                    final String createVoucher = collectionsUtil.getAppConfigValue(
                            CollectionConstants.MODULE_NAME_COLLECTIONS_CONFIG,
                            CollectionConstants.APPCONFIG_VALUE_CREATEVOUCHER_FOR_REMITTANCE);
                    if (CollectionConstants.YES.equalsIgnoreCase(createVoucher)) {
                        final CVoucherHeader voucherHeaderCard = financialsUtil.createRemittanceVoucher(headerdetails,
                                accountCodeOnlineList, subledgerList);

                        newContraVoucherList.add(voucherHeaderCard);
                        depositedBankAccount = (Bankaccount) persistenceService.find(
                                "from Bankaccount where chartofaccounts.glcode=?", serviceGlCode);
                        if (voucherHeaderCard != null && voucherHeaderCard.getId() != null)
                            createVoucherForChequeCardRemittance(instrumentDepositeMap, voucherWorkflowMsg,
                                    voucherTypeForChequeDDCard, voucherDate, depositedBankAccount, serviceGlCode,
                                    instrumentHeaderListOnline, voucherHeaderCard);
                        else {
                            final EgwStatus statusDeposited = egwStatusDAO.getStatusByModuleAndCode(
                                    CollectionConstants.MODULE_NAME_INSTRUMENTHEADER,
                                    CollectionConstants.INSTRUMENT_DEPOSITED_STATUS);
                            financialsUtil.updateInstrumentHeader(instrumentHeaderListOnline, statusDeposited,
                                    depositedBankAccount);
                        }

                        for (final InstrumentHeader instHead : instrumentHeaderListOnline) {
                            final List<ReceiptHeader> receiptHeaders = findAllByNamedQuery(
                                    CollectionConstants.QUERY_RECEIPTS_BY_INSTRUMENTHEADER_AND_SERVICECODE,
                                    instHead.getId(), serviceDetails.getCode());
                            bankRemittanceList.addAll(receiptHeaders);
                        }
                    }
                }
                // If online amount is present
                if (totalOnlineAmount[i].trim() != null && totalOnlineAmount[i].trim().length() > 0
                        && onlinePaymentGlCode != null) {
                    final StringBuilder onlineQueryBuilder = new StringBuilder(receiptInstrumentQueryString);
                    onlineQueryBuilder.append(serviceNameCondition);
                    onlineQueryBuilder.append(receiptDateCondition);
                    onlineQueryBuilder.append(instrumentStatusCondition);
                    onlineQueryBuilder.append(instrumentTypeCondition);
                    onlineQueryBuilder.append(receiptFundCondition);
                    onlineQueryBuilder.append(receiptDepartmentCondition);

                    final Object arguments[] = new Object[6];

                    arguments[0] = serviceName;
                    arguments[1] = receiptDateArray[i];
                    arguments[2] = status.getId();
                    arguments[3] = CollectionConstants.INSTRUMENTTYPE_ONLINE;
                    arguments[4] = fundCodeArray[i];
                    arguments[5] = departmentCodeArray[i];

                    final List<InstrumentHeader> instrumentHeaderListOnline = persistenceService.findAllBy(
                            onlineQueryBuilder.toString(), arguments);

                    final HashMap<String, Object> headerdetails = new HashMap<String, Object>();

                    if (voucherTypeForChequeDDCard) {
                        headerdetails.put(VoucherConstant.VOUCHERNAME,
                                CollectionConstants.FINANCIAL_RECEIPTS_VOUCHERNAME);
                        headerdetails.put(VoucherConstant.VOUCHERTYPE,
                                CollectionConstants.FINANCIAL_RECEIPTS_VOUCHERTYPE);
                    } else {

                        headerdetails.put(VoucherConstant.VOUCHERNAME,
                                CollectionConstants.FINANCIAL_CONTRATVOUCHER_VOUCHERNAME);
                        headerdetails.put(VoucherConstant.VOUCHERTYPE,
                                CollectionConstants.FINANCIAL_CONTRAVOUCHER_VOUCHERTYPE);
                    }
                    headerdetails.put(VoucherConstant.VOUCHERNAME,
                            CollectionConstants.FINANCIAL_CONTRATVOUCHER_VOUCHERNAME);
                    headerdetails.put(VoucherConstant.VOUCHERTYPE,
                            CollectionConstants.FINANCIAL_CONTRAVOUCHER_VOUCHERTYPE);
                    headerdetails.put(VoucherConstant.DESCRIPTION, CollectionConstants.FINANCIAL_VOUCHERDESCRIPTION);
                    headerdetails.put(VoucherConstant.VOUCHERDATE, voucherDate);
                    headerdetails.put(VoucherConstant.FUNDCODE, fundCodeArray[i]);
                    headerdetails.put(VoucherConstant.DEPARTMENTCODE, departmentCodeArray[i]);
                    headerdetails.put(VoucherConstant.FUNDSOURCECODE, serviceDetails.getFundSource() == null ? null
                            : serviceDetails.getFundSource().getCode());
                    headerdetails.put(VoucherConstant.FUNCTIONARYCODE, serviceDetails.getFunctionary() == null ? null
                            : serviceDetails.getFunctionary().getCode());
                    headerdetails.put(VoucherConstant.MODULEID, CollectionConstants.COLLECTIONS_EG_MODULES_ID);

                    final List<HashMap<String, Object>> accountCodeOnlineList = new ArrayList<HashMap<String, Object>>();
                    final HashMap<String, Object> accountcodedetailsCreditOnlineHashMap = new HashMap<String, Object>();

                    accountcodedetailsCreditOnlineHashMap.put(VoucherConstant.GLCODE, onlinePaymentGlCode);
                    accountcodedetailsCreditOnlineHashMap.put(VoucherConstant.FUNCTIONCODE, null);
                    accountcodedetailsCreditOnlineHashMap.put(VoucherConstant.CREDITAMOUNT, totalOnlineAmount[i]);
                    accountcodedetailsCreditOnlineHashMap.put(VoucherConstant.DEBITAMOUNT, 0);

                    accountCodeOnlineList.add(accountcodedetailsCreditOnlineHashMap);
                    // TODO: Add debit account details
                    {
                        final HashMap<String, Object> accountcodedetailsDebitHashMap = new HashMap<String, Object>();
                        accountcodedetailsDebitHashMap.put(VoucherConstant.GLCODE, serviceGlCode);
                        accountcodedetailsDebitHashMap.put(VoucherConstant.FUNCTIONCODE, null);
                        accountcodedetailsDebitHashMap.put(VoucherConstant.CREDITAMOUNT, 0);
                        accountcodedetailsDebitHashMap.put(VoucherConstant.DEBITAMOUNT, totalOnlineAmount[i]);
                        accountCodeOnlineList.add(accountcodedetailsDebitHashMap);
                    }

                    final CVoucherHeader voucherHeaderCard = financialsUtil.createRemittanceVoucher(headerdetails,
                            accountCodeOnlineList, subledgerList);
                    newContraVoucherList.add(voucherHeaderCard);

                    depositedBankAccount = (Bankaccount) persistenceService.find(
                            "from Bankaccount where chartofaccounts.glcode=?", serviceGlCode);
                    if (voucherHeaderCard != null && voucherHeaderCard.getId() != null)
                        createVoucherForChequeCardRemittance(instrumentDepositeMap, voucherWorkflowMsg,
                                voucherTypeForChequeDDCard, voucherDate, depositedBankAccount, serviceGlCode,
                                instrumentHeaderListOnline, voucherHeaderCard);
                    else {
                        final EgwStatus statusDeposited = egwStatusDAO.getStatusByModuleAndCode(
                                CollectionConstants.MODULE_NAME_INSTRUMENTHEADER,
                                CollectionConstants.INSTRUMENT_DEPOSITED_STATUS);
                        financialsUtil.updateInstrumentHeader(instrumentHeaderListOnline, statusDeposited,
                                depositedBankAccount);
                    }

                    for (final InstrumentHeader instHead : instrumentHeaderListOnline) {
                        final List<ReceiptHeader> receiptHeaders = findAllByNamedQuery(
                                CollectionConstants.QUERY_RECEIPTS_BY_INSTRUMENTHEADER_AND_SERVICECODE,
                                instHead.getId(), serviceDetails.getCode());
                        bankRemittanceList.addAll(receiptHeaders);
                    }

                }
            }
        }
        for (final ReceiptHeader receiptHeader : bankRemittanceList) {
            final EgwStatus statusRemitted = egwStatusDAO.getStatusByModuleAndCode(
                    CollectionConstants.MODULE_NAME_RECEIPTHEADER, CollectionConstants.RECEIPT_STATUS_CODE_REMITTED);
            receiptHeader.setStatus(statusRemitted);
            getSession().flush();
            persistenceService.update(receiptHeader);
        }

        return bankRemittanceList;
    }

    private void createVoucherForChequeCardRemittance(final Map<String, Object> instrumentDepositeMap,
            final String voucherWorkflowMsg, final Boolean voucherTypeForChequeDDCard, final Date voucherDate,
            final Bankaccount depositedBankAccount, final String serviceGlCode,
            final List<InstrumentHeader> instrumentHeaderListCheque, final CVoucherHeader voucherHeaderCheque) {
        for (final InstrumentHeader instrumentHeader : instrumentHeaderListCheque)
            if (voucherHeaderCheque.getId() != null && serviceGlCode != null) {
                final Map<String, Object> chequeMap = constructInstrumentMap(instrumentDepositeMap,
                        depositedBankAccount, instrumentHeader, voucherHeaderCheque, voucherDate);
                if (voucherTypeForChequeDDCard)
                    financialsUtil.updateCheque_DD_Card_Deposit_Receipt(voucherHeaderCheque.getId(), serviceGlCode,
                            instrumentHeader, chequeMap);
                else {

                    financialsUtil.updateCheque_DD_Card_Deposit(voucherHeaderCheque.getId(), serviceGlCode,
                            instrumentHeader, chequeMap);
                    final ContraJournalVoucher contraJournalVoucher = (ContraJournalVoucher) persistenceService
                            .findByNamedQuery(CollectionConstants.QUERY_GET_CONTRAVOUCHERBYVOUCHERHEADERID,
                                    voucherHeaderCheque.getId(), instrumentHeader.getId());
                    contraJournalVoucher.transition(true).start()
                            .withSenderName(contraJournalVoucher.getCreatedBy().getName())
                            .withComments(CollectionConstants.WF_STATE_NEW)
                            .withOwner(collectionsUtil.getPositionOfUser(contraJournalVoucher.getCreatedBy()));
                    contraJournalVoucher.transition(true).withSenderName(contraJournalVoucher.getCreatedBy().getName())
                            .withComments(voucherWorkflowMsg)
                            .withOwner(collectionsUtil.getPositionOfUser(contraJournalVoucher.getCreatedBy()));
                }
            }

    }

    private void createVoucherForCashRemittance(final Map<String, Object> instrumentDepositeMap,
            final String voucherWorkflowMsg, final Date voucherDate, final Bankaccount depositedBankAccount,
            final String serviceGlCode, final List<InstrumentHeader> instrumentHeaderListCash,
            final CVoucherHeader voucherHeaderCash) {
        for (final InstrumentHeader instrumentHeader : instrumentHeaderListCash)
            if (voucherHeaderCash.getId() != null && serviceGlCode != null) {
                final Map<String, Object> cashMap = constructInstrumentMap(instrumentDepositeMap, depositedBankAccount,
                        instrumentHeader, voucherHeaderCash, voucherDate);
                financialsUtil.updateCashDeposit(voucherHeaderCash.getId(), serviceGlCode, instrumentHeader, cashMap);
                final ContraJournalVoucher contraJournalVoucher = (ContraJournalVoucher) persistenceService
                        .findByNamedQuery(CollectionConstants.QUERY_GET_CONTRAVOUCHERBYVOUCHERHEADERID,
                                voucherHeaderCash.getId(), instrumentHeader.getId());
                contraJournalVoucher.transition(true).start()
                .withSenderName(contraJournalVoucher.getCreatedBy().getName()).withComments("Voucher Created")
                .withOwner(collectionsUtil.getPositionOfUser(contraJournalVoucher.getCreatedBy()));
                contraJournalVoucher.transition(true).transition()
                .withSenderName(contraJournalVoucher.getCreatedBy().getName()).withComments(voucherWorkflowMsg)
                .withOwner(collectionsUtil.getPositionOfUser(contraJournalVoucher.getCreatedBy()));
            }
    }

    /**
     * For Bill Based Receipt, aggregate the amount for same account heads
     *
     * @param receiptDetailSetParam
     * @return Set of Receipt Detail after Aggregating Amounts
     */
    public Set<ReceiptDetail> aggregateDuplicateReceiptDetailObject(final List<ReceiptDetail> receiptDetailSetParam) {
        final List<ReceiptDetail> newReceiptDetailList = new ArrayList<ReceiptDetail>();

        int counter = 0;

        for (final ReceiptDetail receiptDetailObj : receiptDetailSetParam) {
            if (counter == 0)
                newReceiptDetailList.add(receiptDetailObj);
            else {
                final int checknew = checkIfReceiptDetailObjectExist(newReceiptDetailList, receiptDetailObj);
                if (checknew == -1)
                    newReceiptDetailList.add(receiptDetailObj);
                else {
                    final ReceiptDetail receiptDetail = new ReceiptDetail();

                    final ReceiptDetail newReceiptDetailObj = newReceiptDetailList.get(checknew);
                    newReceiptDetailList.remove(checknew);

                    receiptDetail.setAccounthead(newReceiptDetailObj.getAccounthead());
                    receiptDetail.setAccountPayeeDetails(newReceiptDetailObj.getAccountPayeeDetails());
                    receiptDetail.setCramount(newReceiptDetailObj.getCramount().add(receiptDetailObj.getCramount()));
                    receiptDetail.setCramountToBePaid(newReceiptDetailObj.getCramountToBePaid());
                    receiptDetail.setDescription(newReceiptDetailObj.getDescription());
                    receiptDetail.setDramount(newReceiptDetailObj.getDramount().add(receiptDetailObj.getDramount()));
                    receiptDetail.setFinancialYear(newReceiptDetailObj.getFinancialYear());
                    receiptDetail.setFunction(newReceiptDetailObj.getFunction());
                    receiptDetail.setOrdernumber(newReceiptDetailObj.getOrdernumber());

                    newReceiptDetailList.add(receiptDetail);
                }
            }
            counter++;
        }
        return new HashSet<ReceiptDetail>(newReceiptDetailList);
    }

    /**
     * API to check if the given receipt detail object already exists in the
     * list passed passed as parameter
     *
     * @param newReceiptDetailSet
     * @param receiptDetailObj
     * @return
     */
    public int checkIfReceiptDetailObjectExist(final List<ReceiptDetail> newReceiptDetailSet,
            final ReceiptDetail receiptDetailObj) {
        int check = -1;

        for (int m = 0; m < newReceiptDetailSet.size(); m++) {

            final ReceiptDetail receiptDetail = newReceiptDetailSet.get(m);

            if (receiptDetailObj.getAccounthead().getId().equals(receiptDetail.getAccounthead().getId())) {
                check = m;
                break;
            } else
                continue;
        }
        return check;
    }

    /**
     * End Work-flow of the given cancelled receipt
     *
     * @param receiptHeaders
     *            Set of receipt headers to be transitioned
     * @param actionName
     *            Action name for the transition
     * @param comment
     *            Comment for the transition
     */
    public void endReceiptWorkFlowOnCancellation(final ReceiptHeader receiptHeaderToBeCancelled) {
        // End work-flow for the cancelled receipt
        Position position = null;
        if (!collectionsUtil.isEmployee(receiptHeaderToBeCancelled.getCreatedBy())) {
            final Department dept = departmentService.getDepartmentByName(collectionsUtil.getDepartmentForWorkFlow());
            final Designation desgn = designationService.getDesignationByName(collectionsUtil
                    .getDesignationForThirdPartyUser());
            position = collectionsUtil.getPositionByDeptDesgAndBoundary(dept, desgn, receiptHeaderToBeCancelled
                    .getReceiptMisc().getBoundary());
        } else
            position = collectionsUtil.getPositionOfUser(receiptHeaderToBeCancelled.getCreatedBy());

        if (position != null)
            receiptHeaderToBeCancelled.transition(true).end()
            .withSenderName(receiptHeaderToBeCancelled.getCreatedBy().getName())
            .withComments("Receipt Cancelled - Workflow ends").withStateValue(CollectionConstants.WF_STATE_END)
            .withOwner(position).withDateInfo(new Date());
    }

    /**
     * This method persists the given <code>ReceiptPayeeDetails</code> entity.
     * The receipt number for all of the receipts is generated, if not already
     * present. If the receipt is associated with a challan, and the challan
     * number is not present, the challan number is generated and set into it.
     */
    @Override
    public ReceiptHeader persist(final ReceiptHeader receiptHeader) {
        // for (ReceiptHeader receiptHeader : entity.getReceiptHeaders()) {
        if (receiptHeader.getReceipttype() != CollectionConstants.RECEIPT_TYPE_CHALLAN
                && !CollectionConstants.RECEIPT_STATUS_CODE_PENDING.equals(receiptHeader.getStatus().getCode())
                && receiptHeader.getReceiptnumber() == null)
            setReceiptNumber(receiptHeader);

        if (receiptHeader.getChallan() != null) {
            final Challan challan = receiptHeader.getChallan();
            if (challan.getChallanNumber() == null)
                setChallanNumber(challan);

            receiptHeader.setChallan(challan);
            LOGGER.info("Persisted challan with challan number " + challan.getChallanNumber());
        }
        // }
        return super.persist(receiptHeader);
    }

    /**
     * This method persists the given <code>ReceiptPayeeDetails</code> entity.
     * If the receipt number for all of the receipts is generated, if not
     * already present.
     */

    public ReceiptHeader persistChallan(final ReceiptHeader entity) {
        for (final ReceiptHeader receiptHeader : entity.getReceiptHeaders()) {
            Integer.valueOf(collectionsUtil.getAppConfigValue(CollectionConstants.MODULE_NAME_COLLECTIONS_CONFIG,
                    CollectionConstants.APPCONFIG_VALUE_CHALLANVALIDUPTO));

            final Challan challan = receiptHeader.getChallan();

            /*
             * challan.setValidUpto(eisService.getPriorOrAfterWorkingDate(
             * challan.getChallanDate(), validUpto, DATE_ORDER.AFTER));
             */

            if (challan.getCreatedDate() == null)
                challan.setCreatedDate(new Date());

            if (challan.getChallanNumber() == null)
                setChallanNumber(challan);

            challan.setReceiptHeader(receiptHeader);
            receiptHeader.setChallan(challan);

            LOGGER.info("Persisting challan with challan number " + challan.getChallanNumber());
        }

        return super.persist(entity);
    }

    /**
     * This method persists the given set of <code>ReceiptPayeeDetails</code>
     * instances
     *
     * @param entity
     *            a set of <code>ReceiptPayeeDetails</code> instances to be
     *            persisted
     * @return the list of persisted <code>ReceiptPayeeDetails</code> instances
     */

    public List<ReceiptHeader> persist(final Set<ReceiptHeader> entity) {
        final List<ReceiptHeader> saved = new ArrayList<ReceiptHeader>();
        final Iterator iterator = entity.iterator();
        while (iterator.hasNext()) {
            final ReceiptHeader rpd = (ReceiptHeader) iterator.next();
            saved.add(this.persist(rpd));
        }
        return saved;
    }

    /**
     * This method persists the given set of <code>ReceiptPayeeDetails</code>
     * instances with receipt number as Pending
     *
     * @param entity
     *            a set of <code>ReceiptPayeeDetails</code> instances to be
     *            persisted
     * @return the list of persisted <code>ReceiptPayeeDetails</code> instances
     */

    public List<ReceiptHeader> persistPendingReceipts(final ReceiptHeader receiptHeader) {
        final List<ReceiptHeader> saved = new ArrayList<ReceiptHeader>();
        saved.add(super.persist(receiptHeader));
        return saved;
    }

    public void setReceiptNumber(final ReceiptHeader entity) {
        entity.setReceiptnumber(collectionsNumberGenerator.generateReceiptNumber(entity));
    }

    private void setChallanNumber(final Challan challan) {
        final CFinancialYear financialYear = collectionsUtil.getFinancialYearforDate(challan.getCreatedDate());
        challan.setChallanNumber(collectionsNumberGenerator.generateChallanNumber(challan, financialYear));
    }

    public void setCollectionsNumberGenerator(final CollectionsNumberGenerator collectionsNumberGenerator) {
        this.collectionsNumberGenerator = collectionsNumberGenerator;
    }

    private BillingIntegrationService getBillingServiceBean(final String serviceCode) {
        return (BillingIntegrationService) collectionsUtil.getBean(serviceCode
                + CollectionConstants.COLLECTIONS_INTERFACE_SUFFIX);
    }

    /**
     * This method looks up the bean to communicate with the billing system and
     * updates the billing system.
     */
    public Boolean updateBillingSystem(final String serviceCode, final Set<BillReceiptInfo> billReceipts) {
        final BillingIntegrationService billingService = getBillingServiceBean(serviceCode);
        if (billingService == null)
            return false;
        else
            try {
                billingService.updateReceiptDetails(billReceipts);
                return true;
            } catch (final Exception e) {
                final String errMsg = "Exception while updating billing system [" + serviceCode
                        + "] with receipt details!";
                LOGGER.error(errMsg, e);
                throw new ApplicationRuntimeException(errMsg, e);
            }
    }

    /**
     * This method is called for voucher reversal in case of intra-day receipt
     * cancellation.
     */

    public void createReversalVoucher(final ReceiptVoucher receiptVoucher, final String instrumentType) {
        final List<HashMap<String, Object>> reversalVoucherInfoList = new ArrayList();
        final HashMap<String, Object> reversalVoucherInfo = new HashMap();

        if (receiptVoucher.getVoucherheader() != null) {
            reversalVoucherInfo.put(CollectionConstants.FINANCIALS_VOUCHERREVERSAL_ORIGINALVOUCHERID, receiptVoucher
                    .getVoucherheader().getId());
            reversalVoucherInfo.put(CollectionConstants.FINANCIALS_VOUCHERREVERSAL_DATE, new Date());

            if (receiptVoucher.getVoucherheader().getType()
                    .equals(CollectionConstants.FINANCIAL_JOURNALVOUCHER_VOUCHERTYPE)) {

                reversalVoucherInfo.put(CollectionConstants.FINANCIALS_VOUCHERREVERSAL_TYPE,
                        CollectionConstants.FINANCIAL_JOURNALVOUCHER_VOUCHERTYPE);
                reversalVoucherInfo.put(CollectionConstants.FINANCIALS_VOUCHERREVERSAL_NAME,
                        CollectionConstants.FINANCIAL_JOURNALVOUCHER_VOUCHERNAME);
            } else if (receiptVoucher.getVoucherheader().getType()
                    .equals(CollectionConstants.FINANCIAL_RECEIPTS_VOUCHERTYPE)) {
                reversalVoucherInfo.put(CollectionConstants.FINANCIALS_VOUCHERREVERSAL_TYPE,
                        CollectionConstants.FINANCIAL_PAYMENTVOUCHER_VOUCHERTYPE);
                reversalVoucherInfo.put(CollectionConstants.FINANCIALS_VOUCHERREVERSAL_NAME,
                        CollectionConstants.FINANCIAL_PAYMENTVOUCHER_VOUCHERNAME);
            }
        }

        reversalVoucherInfoList.add(reversalVoucherInfo);

        try {
            financialsUtil.getReversalVoucher(reversalVoucherInfoList);
        } catch (final Exception e) {
            LOGGER.error("Receipt Service Exception while creating reversal voucher!", e);
            // TODO: Throw EGovRuntimeException ?
        }
    }

    /**
     * @ Create instrument voucher list from receiptpayeelist and pass it to
     * financialsutil
     *
     * @param receiptPayeeDetails
     * @return void
     */

    public void updateInstrument(final List<CVoucherHeader> voucherHeaderList,
            final List<InstrumentHeader> instrumentHeaderList) {
        final List<Map<String, Object>> instrumentVoucherList = new ArrayList();
        if (voucherHeaderList != null && instrumentHeaderList != null) {
            for (final CVoucherHeader voucherHeader : voucherHeaderList)
                for (final InstrumentHeader instrumentHeader : instrumentHeaderList) {
                    final Map<String, Object> iVoucherMap = new HashMap();
                    iVoucherMap.put(CollectionConstants.FINANCIAL_INSTRUMENTSERVICE_INSTRUMENTHEADEROBJECT,
                            instrumentHeader);
                    iVoucherMap.put(CollectionConstants.FINANCIAL_INSTRUMENTSERVICE_VOUCHERHEADEROBJECT, voucherHeader);
                    instrumentVoucherList.add(iVoucherMap);
                }
            financialsUtil.updateInstrument(instrumentVoucherList);
        }
    }

    public List<InstrumentHeader> createInstrument(final List<InstrumentHeader> instrumentHeaderList) {
        final List<Map<String, Object>> instrumentHeaderMapList = new ArrayList();
        if (instrumentHeaderList != null)
            for (final InstrumentHeader instrumentHeader : instrumentHeaderList) {
                final Map<String, Object> instrumentHeaderMap = new HashMap();
                instrumentHeaderMap.put(CollectionConstants.MAP_KEY_INSTRSERVICE_INSTRUMENTNUMBER,
                        instrumentHeader.getInstrumentNumber());
                instrumentHeaderMap.put(CollectionConstants.MAP_KEY_INSTRSERVICE_INSTRUMENTDATE,
                        instrumentHeader.getInstrumentDate());
                instrumentHeaderMap.put(CollectionConstants.MAP_KEY_INSTRSERVICE_INSTRUMENTAMOUNT,
                        instrumentHeader.getInstrumentAmount());
                instrumentHeaderMap.put(CollectionConstants.MAP_KEY_INSTRSERVICE_INSTRUMENTTYPE, instrumentHeader
                        .getInstrumentType().getType());
                instrumentHeaderMap.put(CollectionConstants.MAP_KEY_INSTRSERVICE_ISPAYCHEQUE,
                        instrumentHeader.getIsPayCheque());
                if (instrumentHeader.getBankId() != null)
                    instrumentHeaderMap.put(CollectionConstants.MAP_KEY_INSTRSERVICE_BANKCODE, instrumentHeader
                            .getBankId().getCode());
                instrumentHeaderMap.put(CollectionConstants.MAP_KEY_INSTRSERVICE_BANKBRANCHNAME,
                        instrumentHeader.getBankBranchName());
                instrumentHeaderMap.put(CollectionConstants.MAP_KEY_INSTRSERVICE_TRANSACTIONNUMBER,
                        instrumentHeader.getTransactionNumber());
                instrumentHeaderMap.put(CollectionConstants.MAP_KEY_INSTRSERVICE_TRANSACTIONDATE,
                        instrumentHeader.getTransactionDate());
                if (instrumentHeader.getBankAccountId() != null)
                    instrumentHeaderMap.put(CollectionConstants.MAP_KEY_INSTRSERVICE_BANKACCOUNTID, instrumentHeader
                            .getBankAccountId().getId());
                instrumentHeaderMapList.add(instrumentHeaderMap);
                // should add bankaccount for bank : key = Bank account id;
                // value = instrumentHeader.getBankAccount.getId()
            }
        return financialsUtil.createInstrument(instrumentHeaderMapList);
    }

    private Map<String, Object> constructInstrumentMap(final Map<String, Object> instrumentDepositeMap,
            final Bankaccount bankaccount, final InstrumentHeader instrumentHeader, final CVoucherHeader voucherHeader,
            final Date voucherDate) {
        final InstrumentType instrumentType = (InstrumentType) persistenceService.find(
                "select it from InstrumentType it,InstrumentHeader ih where " + "ih.instrumentType=it.id and ih.id=?",
                instrumentHeader.getId());
        instrumentDepositeMap.put("instrumentheader", instrumentHeader.getId());
        instrumentDepositeMap.put("bankaccountid", bankaccount.getId());
        instrumentDepositeMap.put("instrumentamount", instrumentHeader.getInstrumentAmount());
        instrumentDepositeMap.put("instrumenttype", instrumentType.getType());
        instrumentDepositeMap.put("depositdate", voucherDate);
        instrumentDepositeMap.put("createdby", voucherHeader.getCreatedBy().getId());
        instrumentDepositeMap.put("ispaycheque", instrumentHeader.getIsPayCheque());
        instrumentDepositeMap.put("payinid", voucherHeader.getId());
        return instrumentDepositeMap;
    }

    public void performWorkflow(final String actionName, final ReceiptHeader receiptHeader, final String remarks) {
        try {
            Position operatorPosition;
            Department department;
            Employee employee = null;
            final Boolean isEmployee = collectionsUtil.isEmployee(receiptHeader.getCreatedBy());
            if (!isEmployee) {
                department = departmentService.getDepartmentByName(collectionsUtil.getDepartmentForWorkFlow());
                final Designation desgn = designationService.getDesignationByName(collectionsUtil
                        .getDesignationForThirdPartyUser());
                employee = employeeService.getEmployeeById(collectionsUtil.getLoggedInUser().getId());
                operatorPosition = collectionsUtil.getPositionByDeptDesgAndBoundary(department, desgn, receiptHeader
                        .getReceiptMisc().getBoundary());
            } else {
                operatorPosition = collectionsUtil.getPositionOfUser(receiptHeader.getCreatedBy());
                department = collectionsUtil.getDepartmentOfUser(receiptHeader.getCreatedBy());
                employee = employeeService.getEmployeeById(receiptHeader.getCreatedBy().getId());
            }
            // TODO Get the Designation name from AppConfig
            final Designation designation = designationService.getDesignationByName("Section manager");
            Boundary boundary = null;
            for (final Jurisdiction jur : employee.getJurisdictions())
                boundary = jur.getBoundary();
            final List<Employee> emp = employeeService.findByDepartmentDesignationAndBoundary(department.getId(),
                    designation.getId(), boundary.getId());
            final Position approverPosition = collectionsUtil.getPositionOfUser(emp.get(0));
            System.out.println(approverPosition.getName());
            if (actionName.equals(CollectionConstants.WF_ACTION_SUBMIT))
                perform(receiptHeader, CollectionConstants.WF_ACTION_APPROVE,
                        CollectionConstants.RECEIPT_STATUS_CODE_SUBMITTED, CollectionConstants.WF_ACTION_APPROVE,
                        approverPosition, remarks);
            else if (actionName.equals(CollectionConstants.WF_ACTION_APPROVE))
                perform(receiptHeader, CollectionConstants.WF_STATE_APPROVED,
                        CollectionConstants.RECEIPT_STATUS_CODE_APPROVED, "", approverPosition, remarks);
            else if (actionName.equals(CollectionConstants.WF_ACTION_REJECT))
                perform(receiptHeader, CollectionConstants.WF_STATE_REJECTED,
                        CollectionConstants.RECEIPT_STATUS_CODE_TO_BE_SUBMITTED, CollectionConstants.WF_ACTION_SUBMIT,
                        operatorPosition, remarks);
        } catch (final ValidationException e) {
            LOGGER.error("Receipt Service Exception while workflow transition!", e);
        }
    }

    public void perform(final ReceiptHeader receiptHeader, final String wfState, final String newStatusCode,
            final String nextAction, final Position ownerPosition, final String remarks) {
        receiptHeader.setStatus(collectionsUtil.getReceiptStatusForCode(newStatusCode));

        if (receiptHeader.getStatus().getCode().equals(CollectionConstants.RECEIPT_STATUS_CODE_APPROVED))
            // Receipt approved. end workflow for this receipt.
            receiptHeader.transition().end().withSenderName(receiptHeader.getCreatedBy().getName())
            .withComments("Receipt Approved - Workflow ends").withStateValue(CollectionConstants.WF_STATE_END)
            .withOwner(ownerPosition).withDateInfo(new Date());
        else
            receiptHeader.transition().withSenderName(receiptHeader.getCreatedBy().getName()).withComments(remarks)
            .withStateValue(wfState).withOwner(ownerPosition).withDateInfo(new Date())
            .withNextAction(nextAction);
        getSession().flush();
        persistenceService.persist(receiptHeader);
    }

    public Set<InstrumentHeader> createOnlineInstrument(final Date transactionDate, final String transactionId,
            final BigDecimal transactionAmt) {
        final InstrumentHeader onlineInstrumentHeader = new InstrumentHeader();
        Set<InstrumentHeader> instrumentHeaderSet = new HashSet<InstrumentHeader>();
        onlineInstrumentHeader.setInstrumentType(financialsUtil
                .getInstrumentTypeByType(CollectionConstants.INSTRUMENTTYPE_ONLINE));
        onlineInstrumentHeader.setTransactionDate(transactionDate);
        onlineInstrumentHeader.setIsPayCheque(CollectionConstants.ZERO_INT);
        onlineInstrumentHeader.setTransactionNumber(transactionId);
        onlineInstrumentHeader.setInstrumentAmount(transactionAmt);

        final List<InstrumentHeader> instHeaderList = new ArrayList<InstrumentHeader>();
        instHeaderList.add(onlineInstrumentHeader);
        instrumentHeaderSet = new HashSet(createInstrument(instHeaderList));
        return instrumentHeaderSet;
    }

    public String getReceiptHeaderforDishonor(final Long mode, final Long bankAccId, final Long bankId,
            final String chequeDDNo, final String chqueDDDate) {
        final StringBuilder sb = new StringBuilder(300);
        new ArrayList<Object>();
        sb.append("FROM egcl_collectionheader rpt,egcl_collectioninstrument ci,egf_instrumentheader ih,egw_status status,bank b,"
                + "bankbranch bb,bankaccount ba WHERE rpt.id = ci.collectionheader AND ci.instrumentheader = ih.id AND status.id = ih.id_status "
                + "AND b.id = bb.bankid AND bb.id = ba.branchid AND ba.id = ih.bankaccountid AND ih.instrumenttype = '"
                + mode
                + "' AND ((ih.ispaycheque ='0' AND status.moduletype ='"
                + CollectionConstants.MODULE_NAME_INSTRUMENTHEADER
                + "'"
                + "AND status.description = '"
                + CollectionConstants.INSTRUMENT_DEPOSITED_STATUS + "'))");

        if (bankAccId != null && bankAccId != 0)
            sb.append(" AND ih.bankaccountid=" + bankAccId + "");
        if ((bankAccId == null || bankAccId == 0) && bankId != null && bankId != 0)
            sb.append(" AND ih.bankid=" + bankAccId + "");
        if (!"".equals(chequeDDNo) && chequeDDNo != null)
            sb.append(" AND ih.instrumentnumber=trim('" + chequeDDNo + "') ");
        if (!"".equals(chqueDDDate) && chqueDDDate != null)
            sb.append(" AND ih.instrumentdate >= '" + chqueDDDate + "' ");

        return sb.toString();
    }

    public void setCollectionsUtil(final CollectionsUtil collectionsUtil) {
        this.collectionsUtil = collectionsUtil;
    }

    public void setFinancialsUtil(final FinancialsUtil financialsUtil) {
        this.financialsUtil = financialsUtil;
    }

    public void setPersistenceService(final PersistenceService persistenceService) {
        this.persistenceService = persistenceService;
    }
}
